/*
 * Decompiled with CFR 0.152.
 */
package net.filebot;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import net.filebot.RenameAction;
import net.filebot.UserFiles;
import net.filebot.util.FileUtilities;

public enum StandardRenameAction implements RenameAction
{
    MOVE{

        @Override
        public File rename(File from, File to) throws Exception {
            return FileUtilities.moveRename(from, to);
        }
    }
    ,
    COPY{

        @Override
        public File rename(File from, File to) throws Exception {
            return FileUtilities.copyAs(from, to);
        }
    }
    ,
    KEEPLINK{

        @Override
        public File rename(File from, File to) throws Exception {
            File dest = FileUtilities.resolveDestination(from, to);
            try {
                Files.move(from.toPath(), dest.toPath(), new CopyOption[0]);
                FileUtilities.createRelativeSymlink(from, dest, true);
            }
            catch (LinkageError e) {
                throw new Exception("Unsupported Operation: move, createSymbolicLink");
            }
            return dest;
        }
    }
    ,
    SYMLINK{

        @Override
        public File rename(File from, File to) throws Exception {
            File dest = FileUtilities.resolveDestination(from, to);
            try {
                return FileUtilities.createRelativeSymlink(dest, from, true);
            }
            catch (LinkageError e) {
                throw new Exception("Unsupported Operation: createSymbolicLink");
            }
        }
    }
    ,
    HARDLINK{

        @Override
        public File rename(File from, File to) throws Exception {
            File dest = FileUtilities.resolveDestination(from, to);
            try {
                return FileUtilities.createHardLinkStructure(dest, from);
            }
            catch (LinkageError e) {
                throw new Exception("Unsupported Operation: createLink");
            }
        }
    }
    ,
    DUPLICATE{

        @Override
        public File rename(File from, File to) throws Exception {
            try {
                return HARDLINK.rename(from, to);
            }
            catch (Exception e) {
                return COPY.rename(from, to);
            }
        }
    }
    ,
    REFLINK{

        @Override
        public File rename(File from, File to) throws Exception {
            File dest = FileUtilities.resolveDestination(from, to);
            ProcessBuilder process = new ProcessBuilder("cp", "--reflink", "--force", from.isDirectory() ? "--recursive" : "--no-target-directory", from.getPath(), dest.getPath());
            process.directory(from.getParentFile());
            process.inheritIO();
            int exitCode = process.start().waitFor();
            if (exitCode != 0) {
                throw new IOException(String.format("reflink: %s failed with exit code %d", process.command(), exitCode));
            }
            return dest;
        }
    }
    ,
    TEST{

        @Override
        public File rename(File from, File to) throws IOException {
            return FileUtilities.resolve(from, to);
        }

        @Override
        public boolean canRevert() {
            return false;
        }
    };


    public String getDisplayName() {
        switch (this) {
            case MOVE: {
                return "Rename";
            }
            case COPY: {
                return "Copy";
            }
            case KEEPLINK: {
                return "Keeplink";
            }
            case SYMLINK: {
                return "Symlink";
            }
            case HARDLINK: {
                return "Hardlink";
            }
            case DUPLICATE: {
                return "Hardlink or Copy";
            }
            case REFLINK: {
                return "Lightweight Copy";
            }
        }
        return "Test";
    }

    public String getDisplayVerb() {
        switch (this) {
            case MOVE: {
                return "Moving";
            }
            case COPY: {
                return "Copying";
            }
            case KEEPLINK: {
                return "Moving and symlinking";
            }
            case SYMLINK: {
                return "Symlinking";
            }
            case HARDLINK: {
                return "Hardlinking";
            }
            case DUPLICATE: {
                return "Duplicating";
            }
            case REFLINK: {
                return "Reflinking";
            }
        }
        return "Testing";
    }

    public static List<String> names() {
        return Arrays.stream(StandardRenameAction.values()).map(Enum::name).collect(Collectors.toList());
    }

    public static StandardRenameAction forName(String name) {
        for (StandardRenameAction action : StandardRenameAction.values()) {
            if (!action.name().equalsIgnoreCase(name)) continue;
            return action;
        }
        throw new IllegalArgumentException(String.format("%s not in %s", name, StandardRenameAction.names()));
    }

    public static File revert(File current, File original) throws IOException {
        if (current.equals(original)) {
            return original;
        }
        if (current.exists() && !original.exists()) {
            return FileUtilities.moveRename(current, original);
        }
        BasicFileAttributes currentAttr = Files.readAttributes(current.toPath(), BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
        BasicFileAttributes originalAttr = Files.readAttributes(original.toPath(), BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
        if (currentAttr.isSymbolicLink() && !originalAttr.isSymbolicLink()) {
            UserFiles.trash(current);
            return original;
        }
        if (!currentAttr.isSymbolicLink() && originalAttr.isSymbolicLink()) {
            UserFiles.trash(original);
            return FileUtilities.moveRename(current, original);
        }
        if (currentAttr.isRegularFile() && originalAttr.isRegularFile()) {
            UserFiles.trash(current);
            return original;
        }
        if (currentAttr.isDirectory() && originalAttr.isDirectory()) {
            UserFiles.trash(original);
            return FileUtilities.moveRename(current, original);
        }
        throw new IllegalArgumentException(String.format("Cannot revert file: %s => %s", current, original));
    }
}

