/*
 * Decompiled with CFR 0.152.
 */
package net.filebot.web;

import java.net.URI;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.swing.Icon;
import net.filebot.Cache;
import net.filebot.CacheType;
import net.filebot.CachedResource;
import net.filebot.Logging;
import net.filebot.ResourceManager;
import net.filebot.util.JsonUtilities;
import net.filebot.util.StringUtilities;
import net.filebot.web.AbstractEpisodeListProvider;
import net.filebot.web.Artwork;
import net.filebot.web.ArtworkProvider;
import net.filebot.web.Episode;
import net.filebot.web.EpisodeInfo;
import net.filebot.web.EpisodeUtilities;
import net.filebot.web.Person;
import net.filebot.web.SearchResult;
import net.filebot.web.SeriesInfo;
import net.filebot.web.SimpleDate;
import net.filebot.web.SortOrder;
import net.filebot.web.TheTVDBSeriesInfo;
import net.filebot.web.WebRequest;

public class TheTVDBClient
extends AbstractEpisodeListProvider
implements ArtworkProvider {
    private String apikey;
    private String token = null;
    private Instant tokenExpireInstant = null;
    private Duration tokenExpireDuration = Duration.ofHours(1L);

    public TheTVDBClient(String apikey) {
        this.apikey = apikey;
    }

    @Override
    public String getIdentifier() {
        return "TheTVDBv2";
    }

    @Override
    public Icon getIcon() {
        return ResourceManager.getIcon("search.thetvdb");
    }

    @Override
    public boolean hasSeasonSupport() {
        return true;
    }

    protected Object postJson(String path, Object object) throws Exception {
        ByteBuffer response = WebRequest.post(this.getEndpoint(path), JsonUtilities.json(object, false).getBytes(StandardCharsets.UTF_8), "application/json", null);
        return JsonUtilities.readJson(StandardCharsets.UTF_8.decode(response));
    }

    protected Object requestJson(String path, Locale locale, Duration expirationTime) throws Exception {
        Cache cache = Cache.getCache(locale == null || locale == Locale.ROOT ? this.getName() : this.getName() + "_" + locale.getLanguage(), CacheType.Monthly);
        return cache.json(path, this::getEndpoint).fetch(CachedResource.fetchIfModified(() -> this.getRequestHeader(locale))).expire(expirationTime).get();
    }

    protected URL getEndpoint(String path) throws Exception {
        return new URL("https://api.thetvdb.com/" + path);
    }

    private Map<String, String> getRequestHeader(Locale locale) {
        LinkedHashMap<String, String> header = new LinkedHashMap<String, String>(3);
        if (locale != null && locale != Locale.ROOT) {
            header.put("Accept-Language", Stream.of(locale, Locale.ENGLISH).map(Locale::getLanguage).distinct().collect(Collectors.joining(", ")));
        }
        header.put("Accept", "application/json");
        header.put("Authorization", "Bearer " + this.getAuthorizationToken());
        return header;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String getAuthorizationToken() {
        Duration duration = this.tokenExpireDuration;
        synchronized (duration) {
            if (this.token == null || this.tokenExpireInstant != null && Instant.now().isAfter(this.tokenExpireInstant)) {
                try {
                    Object json = this.postJson("login", Collections.singletonMap("apikey", this.apikey));
                    this.token = JsonUtilities.getString(json, "token");
                    this.tokenExpireInstant = Instant.now().plus(this.tokenExpireDuration);
                }
                catch (Exception e) {
                    throw new IllegalStateException("Failed to retrieve authorization token: " + e.getMessage(), e);
                }
            }
            return this.token;
        }
    }

    protected List<SearchResult> search(String path, Map<String, Object> query, Locale locale, Duration expirationTime) throws Exception {
        Object json = this.requestJson(path + "?" + WebRequest.encodeParameters(query, true), locale, expirationTime);
        return JsonUtilities.streamJsonObjects(json, "data").map(it -> {
            int id = JsonUtilities.getInteger(it, "id");
            String seriesName = JsonUtilities.getString(it, "seriesName");
            String[] aliasNames = (String[])Arrays.stream(JsonUtilities.getArray(it, "aliases")).toArray(String[]::new);
            if (seriesName.startsWith("**") && seriesName.endsWith("**")) {
                Logging.debug.fine(Logging.format("Invalid series: %s [%d]", seriesName, id));
                return null;
            }
            return new SearchResult(id, seriesName, aliasNames);
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    @Override
    public List<SearchResult> fetchSearchResult(String query, Locale locale) throws Exception {
        return this.search("search/series", Collections.singletonMap("name", query), locale, Cache.ONE_DAY);
    }

    @Override
    public TheTVDBSeriesInfo getSeriesInfo(int id, Locale language) throws Exception {
        return this.getSeriesInfo(new SearchResult(id), language);
    }

    @Override
    public TheTVDBSeriesInfo getSeriesInfo(SearchResult series, Locale locale) throws Exception {
        Object json = this.requestJson("series/" + series.getId(), locale, Cache.ONE_WEEK);
        Map<?, ?> data = JsonUtilities.getMap(json, "data");
        TheTVDBSeriesInfo info = new TheTVDBSeriesInfo(this, locale, series.getId());
        info.setAliasNames((String[])Stream.of(series.getAliasNames(), JsonUtilities.getArray(data, "aliases")).flatMap(it -> Arrays.stream(it)).map(Object::toString).distinct().toArray(String[]::new));
        info.setName(JsonUtilities.getString(data, "seriesName"));
        info.setCertification(JsonUtilities.getString(data, "rating"));
        info.setNetwork(JsonUtilities.getString(data, "network"));
        info.setStatus(JsonUtilities.getString(data, "status"));
        info.setRating(JsonUtilities.getDecimal(data, "siteRating"));
        info.setRatingCount(JsonUtilities.getInteger(data, "siteRatingCount"));
        info.setRuntime(StringUtilities.matchInteger(JsonUtilities.getString(data, "runtime")));
        info.setGenres(Arrays.stream(JsonUtilities.getArray(data, "genre")).map(Object::toString).collect(Collectors.toList()));
        info.setStartDate(JsonUtilities.getStringValue(data, "firstAired", SimpleDate::parse));
        info.setImdbId(JsonUtilities.getString(data, "imdbId"));
        info.setOverview(JsonUtilities.getString(data, "overview"));
        info.setAirsDayOfWeek(JsonUtilities.getString(data, "airsDayOfWeek"));
        info.setAirsTime(JsonUtilities.getString(data, "airsTime"));
        info.setBannerUrl(JsonUtilities.getStringValue(data, "banner", this::resolveImage));
        info.setLastUpdated(JsonUtilities.getStringValue(data, "lastUpdated", Long::new));
        return info;
    }

    @Override
    protected AbstractEpisodeListProvider.SeriesData fetchSeriesData(SearchResult series, SortOrder sortOrder, Locale locale) throws Exception {
        TheTVDBSeriesInfo info = this.getSeriesInfo(series, locale);
        info.setOrder(sortOrder.name());
        ArrayList<Episode> episodes = new ArrayList<Episode>();
        ArrayList specials = new ArrayList();
        int n = 1;
        for (int i = 1; i <= n; ++i) {
            Object json = this.requestJson("series/" + series.getId() + "/episodes?page=" + i, locale, Cache.ONE_DAY);
            Integer lastPage = JsonUtilities.getInteger(JsonUtilities.getMap(json, "links"), "last");
            if (lastPage != null) {
                n = lastPage;
            }
            JsonUtilities.streamJsonObjects(json, "data").forEach(it -> {
                String episodeName = JsonUtilities.getString(it, "episodeName");
                Integer absoluteNumber = JsonUtilities.getInteger(it, "absoluteNumber");
                SimpleDate airdate = JsonUtilities.getStringValue(it, "firstAired", SimpleDate::parse);
                Integer id = JsonUtilities.getInteger(it, "id");
                Integer episodeNumber = JsonUtilities.getInteger(it, "airedEpisodeNumber");
                Integer seasonNumber = JsonUtilities.getInteger(it, "airedSeason");
                if (sortOrder == SortOrder.DVD) {
                    Integer dvdSeasonNumber = JsonUtilities.getInteger(it, "dvdSeason");
                    Integer dvdEpisodeNumber = JsonUtilities.getInteger(it, "dvdEpisodeNumber");
                    if (dvdSeasonNumber != null && dvdEpisodeNumber != null) {
                        seasonNumber = dvdSeasonNumber;
                        episodeNumber = dvdEpisodeNumber;
                    }
                } else if (sortOrder == SortOrder.Absolute && absoluteNumber != null && absoluteNumber > 0) {
                    seasonNumber = null;
                    episodeNumber = absoluteNumber;
                } else if (sortOrder == SortOrder.AbsoluteAirdate && airdate != null) {
                    seasonNumber = null;
                    episodeNumber = airdate.getYear() * 10000 + airdate.getMonth() * 100 + airdate.getDay();
                }
                if (seasonNumber == null || seasonNumber > 0) {
                    episodes.add(new Episode(info.getName(), seasonNumber, episodeNumber, episodeName, absoluteNumber, null, airdate, id, new SeriesInfo(info)));
                } else {
                    specials.add(new Episode(info.getName(), null, null, episodeName, absoluteNumber, episodeNumber, airdate, id, new SeriesInfo(info)));
                }
            });
        }
        episodes.sort(EpisodeUtilities.episodeComparator());
        episodes.addAll(specials);
        return new AbstractEpisodeListProvider.SeriesData(info, episodes);
    }

    public SearchResult lookupByID(int id, Locale locale) throws Exception {
        if (id <= 0) {
            throw new IllegalArgumentException("Illegal TheTVDB ID: " + id);
        }
        TheTVDBSeriesInfo info = this.getSeriesInfo(new SearchResult(id), locale);
        return new SearchResult(id, info.getName(), info.getAliasNames());
    }

    public SearchResult lookupByIMDbID(int imdbid, Locale locale) throws Exception {
        if (imdbid <= 0) {
            throw new IllegalArgumentException("Illegal IMDbID ID: " + imdbid);
        }
        List<SearchResult> result = this.search("search/series", Collections.singletonMap("imdbId", String.format("tt%07d", imdbid)), locale, Cache.ONE_MONTH);
        return result.size() > 0 ? result.get(0) : null;
    }

    @Override
    public URI getEpisodeListLink(SearchResult searchResult) {
        return URI.create("http://www.thetvdb.com/?tab=seasonall&id=" + searchResult.getId());
    }

    @Override
    public List<Artwork> getArtwork(int id, String category, Locale locale) throws Exception {
        Object json = this.requestJson("series/" + id + "/images/query?keyType=" + category, locale, Cache.ONE_MONTH);
        return JsonUtilities.streamJsonObjects(json, "data").map(it -> {
            String subKey = JsonUtilities.getString(it, "subKey");
            String resolution = JsonUtilities.getString(it, "resolution");
            URL url = JsonUtilities.getStringValue(it, "fileName", this::resolveImage);
            Double rating = JsonUtilities.getDecimal(JsonUtilities.getMap(it, "ratingsInfo"), "average");
            return new Artwork(Stream.of(category, subKey, resolution), url, locale, rating);
        }).sorted(Artwork.RATING_ORDER).collect(Collectors.toList());
    }

    protected URL resolveImage(String path) {
        if (path == null || path.isEmpty()) {
            return null;
        }
        try {
            return new URL("http://thetvdb.com/banners/" + path);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(path, e);
        }
    }

    public List<String> getLanguages() throws Exception {
        Object response = this.requestJson("languages", Locale.ROOT, Cache.ONE_MONTH);
        return JsonUtilities.streamJsonObjects(response, "data").map(it -> JsonUtilities.getString(it, "abbreviation")).collect(Collectors.toList());
    }

    public List<Person> getActors(int seriesId, Locale locale) throws Exception {
        Object response = this.requestJson("series/" + seriesId + "/actors", locale, Cache.ONE_MONTH);
        return JsonUtilities.streamJsonObjects(response, "data").map(it -> {
            String name = JsonUtilities.getString(it, "name");
            String character = JsonUtilities.getString(it, "role");
            Integer order = JsonUtilities.getInteger(it, "sortOrder");
            URL image = JsonUtilities.getStringValue(it, "image", this::resolveImage);
            return new Person(name, character, "Actor", null, order, image);
        }).sorted(Person.CREDIT_ORDER).collect(Collectors.toList());
    }

    public EpisodeInfo getEpisodeInfo(int id, Locale locale) throws Exception {
        Object response = this.requestJson("episodes/" + id, locale, Cache.ONE_MONTH);
        Map<?, ?> data = JsonUtilities.getMap(response, "data");
        Integer seriesId = JsonUtilities.getInteger(data, "seriesId");
        String overview = JsonUtilities.getString(data, "overview");
        Double rating = JsonUtilities.getDecimal(data, "siteRating");
        Integer votes = JsonUtilities.getInteger(data, "siteRatingCount");
        ArrayList<Person> people = new ArrayList<Person>();
        for (Object it : JsonUtilities.getArray(data, "directors")) {
            people.add(new Person(it.toString(), "Director"));
        }
        for (Object it : JsonUtilities.getArray(data, "writers")) {
            people.add(new Person(it.toString(), "Writer"));
        }
        for (Object it : JsonUtilities.getArray(data, "guestStars")) {
            people.add(new Person(it.toString(), "Guest Star"));
        }
        return new EpisodeInfo(this, locale, seriesId, id, people, overview, rating, votes);
    }
}

