#include "util.h"
#include <iostream>
#include <fstream>
#include <streambuf>

//location in PS2 virtual address space, used for track lookup table
//this is where the song name table will be placed in memory
unsigned int BGM::getPatchOffset()
{
	return 0x009ACE10;
}

const unsigned int ORIG_TRACK_POINTER_COUNT = 116;
const char* ORIG_TRACK_POINTER_ORDER[ORIG_TRACK_POINTER_COUNT] = {
	"01.ADX", "01.ADX", "26.ADX", NULL, "26.ADX", "26.ADX", "26.ADX", NULL, "26.ADX", NULL, "26.ADX", 
	NULL, "26.ADX", NULL, "26.ADX", NULL, "26.ADX", NULL, "26.ADX", "19.ADX", "20.ADX", "21.ADX", "22.ADX",
	"23.ADX", "24.ADX", "25.ADX", "26.ADX", "27.ADX", "28.ADX", "29.ADX", "30.ADX", "31.ADX", "32.ADX", 
	"26.ADX", "26.ADX", "35.ADX", "36.ADX", "37.ADX", "38.ADX", "39.ADX", "40.ADX", "THEME.ADX", "THEME.ADX",
	"43.ADX", "44.ADX", "45.ADX", "46.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "50.ADX", "51.ADX", "52.ADX",
	"53.ADX", "54.ADX", "55.ADX", "56.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "60.ADX", "61.ADX", "62.ADX",
	"63.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "THEME.ADX", "70.ADX", "71.ADX", 
	"72.ADX", "73.ADX", "74.ADX", "75.ADX", "76.ADX", "77.ADX", "78.ADX", "79.ADX", "80.ADX", "81.ADX", 
	"82.ADX", "83.ADX", "84.ADX", "85.ADX", "86.ADX", "87.ADX", "88.ADX", "89.ADX", "90.ADX", "91.ADX", 
	"92.ADX", "93.ADX", "94.ADX", "95.ADX", "96.ADX", "97.ADX", "98.ADX", "99.ADX", "100.ADX", "101.ADX", 
	"102.ADX", "103.ADX", "104.ADX", "105.ADX", "106.ADX", "107.ADX", "108.ADX", "109.ADX",
	"110.ADX", "111.ADX", "112.ADX", "113.ADX", "114.ADX", "115.ADX"
};

BGM::BGM(std::string origPath, std::string addPath, std::string outPath)
{
	mOrigPath = origPath;
	mAddPath = addPath;
	mOutPath = outPath;
	load();
}

Track* BGM::getTrackById(unsigned int trackId)
{
	if(ORIG_TRACK_POINTER_ORDER[trackId] == NULL)
		return NULL;

	for(unsigned int i = 0; i < mTracks.size(); i++)
	{
		if(mTracks.at(i).getOrigPath().filename().string() == ORIG_TRACK_POINTER_ORDER[trackId])
		{
			return &mTracks.at(i);
		}
	}

	std::cout << "Track for ID " << trackId << " not found! Is your 'original' folder correct?\n";
	return NULL;
}

bool BGM::isTrack(std::string filename)
{
	for(unsigned int i = 0; i < ORIG_TRACK_POINTER_COUNT; i++)
	{
		if(ORIG_TRACK_POINTER_ORDER[i] == NULL)
			continue;

		if(filename == ORIG_TRACK_POINTER_ORDER[i])
		{
			return true;
		}
	}

	return false;
}

void BGM::load()
{
	if(!fs::is_directory(mOrigPath))
	{
		std::cout << "Missing original path!\n";
		return;
	}

	if(!fs::is_directory(mAddPath))
	{
		std::cout << "Missing additional path!\n";
		return;
	}

	//load original files
	fs::directory_iterator endIter;
	for(fs::directory_iterator i(mOrigPath); i != endIter; i++)
	{
		fs::path p = (*i).path();

		if(isTrack(p.filename().string()))
		{
			//it's a valid track file
			mTracks.push_back(Track(p));
		}else{
			//it's something else we don't need to worry about (but will copy over to our 'out' folder eventually)
			mOtherFiles.push_back(p);
		}
	}

	//load new files
	for(unsigned int i = 0; i < mTracks.size(); i++)
	{
		Track* t = &mTracks.at(i);

		fs::path addTrackFolder(mAddPath);
		addTrackFolder /= t->getOrigPath().stem();

		if(fs::exists(addTrackFolder))
		{
			if(!fs::is_directory(addTrackFolder))
			{
				std::cout << addTrackFolder.string() << " exists, but is not a folder! Skipping.\n";
				continue;
			}

			for(fs::directory_iterator i(addTrackFolder); i != endIter; i++)
			{
				fs::path p = (*i).path();
				if(p.extension() == ".ADX" || p.extension() == ".adx")
					t->addSong(p);
			}
		}
	}
}

void BGM::print()
{
	std::cout << "Track count: " << mTracks.size() << "\n";

	for(unsigned int i = 0; i < mTracks.size(); i++)
	{
		Track* track = &mTracks.at(i);
		std::cout << "-" << track->getOrigPath().filename().string() << "\n";

		for(unsigned char s = 1; s < track->getSongCount(); s++)
		{
			std::cout << "   +" << track->getSongPath(s).filename().string() << "\n";
		}
	}

	std::cout << "------------------\n";
}

char* BGM::packSongFilename(Track* track, unsigned char id, unsigned int* lengthOut)
{
	*lengthOut = track->getSongDataSize();
	
	char* out = new char[*lengthOut];
	
	unsigned int i = 0;
	std::string filename = track->getSongFilename(id);
	while(i < filename.length())
	{
		out[i] = filename[i];
		i++;
	}

	while(i < *lengthOut)
	{
		out[i] = '\0';
		i++;
	}

	return out;
}

unsigned int BGM::getPathAddress(unsigned int trackIndex, unsigned int offset)
{
	unsigned int addr = 0;

	for(unsigned int i = 0; i < trackIndex; i++)
	{
		Track* t = getTrackById(i);
		if(t == NULL)
			continue;

		addr += t->getSongDataSize() * t->getSongCount();
	}

	return addr + offset;
}

//we write the data out to a file as an intermediate step;
//it's merged into the executable with the Patcher class
void BGM::writeSongNameTable(const std::string& outPath)
{
	std::ofstream out(outPath, std::ios::out | std::ios::binary);

	std::cout << "Writing song name table to " << outPath << "...";

	//write song path array
	char* namePacked;
	unsigned int namePackedLength;
	std::string nameStr;

	for(unsigned int t = 0; t < ORIG_TRACK_POINTER_COUNT; t++)
	{
		Track* track = getTrackById(t);

		if(track == NULL)
			continue;

		for(unsigned char s = 0; s < track->getSongCount(); s++)
		{
			namePacked = packSongFilename(track, s, &namePackedLength);
			out.write(namePacked, namePackedLength);
			
			delete[] namePacked;
		}
	}

	std::cout << "done\n";
}

void BGM::writeTrackLookupTable(const std::string& outPath)
{
	std::ofstream out(outPath, std::ios::out | std::ios::binary);

	std::cout << "Writing track lookup table to " << outPath << "...";

	//write track data array
	for(unsigned int i = 0; i < ORIG_TRACK_POINTER_COUNT; i++)
	{
		Track* t = getTrackById(i);
		char trackData[TRACK_DATA_SIZE];
		std::fill(trackData, trackData + TRACK_DATA_SIZE, 0);

		if(t != NULL)
		{
			unsigned int addr = getPathAddress(i, getPatchOffset());
			std::cout << "Writing track " << t->getSongFilename(0) << " (" << std::hex << addr << ")\n";
			*((unsigned int*)trackData) = addr;
			
			char* trackCount = trackData + sizeof(unsigned int);
			*trackCount = t->getSongCount();

			unsigned int* trackSize = (unsigned int*)(trackData + sizeof(unsigned int)*2);
			*trackSize = t->getSongDataSize();

		}else{
			std::cout << "Writing NULL track\n";
		}

		out.write(trackData, TRACK_DATA_SIZE);
	}

	out.close();

	std::cout << "done\n";
}

void BGM::buildDirectory()
{
	fs::path outPath(mOutPath);

	std::cout << "Building final directory in \"" << mOutPath << "\"...\n";

	//warn if potentially overwriting files in out directory
	if(fs::is_directory(outPath))
	{
		std::cout << "Warning - folder \"" << mOutPath << "\" already exists. Files within may be overwritten. Ok, Y/N? ";
		std::string okStr;
		std::cin >> okStr;
		if(okStr.length() == 0 || (okStr[0] != 'Y' && okStr[0] != 'y'))
		{
			std::cout << "Aborting.\n";
			return;
		}
	}else{
		fs::create_directory(outPath);
	}

	std::cout << "   Copying non-track files...";
	for(unsigned int i = 0; i < mOtherFiles.size(); i++)
	{
		fs::path path = mOtherFiles.at(i);
		fs::copy_file(path, outPath / path.filename(), fs::copy_option::overwrite_if_exists);
	}
	std::cout << "done\n";

	std::cout << "   Copying and naming track files...";
	for(unsigned int i = 0; i < mTracks.size(); i++)
	{
		Track* track = &mTracks.at(i);

		fs::copy_file(track->getOrigPath(), outPath / track->getOrigPath().filename(), fs::copy_option::overwrite_if_exists);

		for(unsigned char s = 1; s < track->getSongCount(); s++)
		{
			std::string nameStr = track->getSongFilename(s);
			fs::copy_file(track->getSongPath(s), outPath / nameStr, fs::copy_option::overwrite_if_exists);
		}
	}
	std::cout << "done\n";
}

Track::Track(fs::path origPath)
{
	mSongs.push_back(origPath);
}
