#include <iostream>
#include <boost/filesystem.hpp>
#include "util.h"
#include "Patcher.h"
#include <fstream>
#include "iso/iso_parser.h"
#include "FileTable.h"

#define PS2OFFSET 0x000fff80 //this is the offset between a hex address and a PS2 memory address; add to hex address or subtract from PS2 address to convert
#define ELF_FOOTER_OFFSET 0x00000180 //this offset was found through experimentation; as such, it's probably not 100% correct...but it works!

#define ROUTINE_MEMORY_POS 0x005d4014 //overwrite old track lookup table
#define ROUTINE_JUMP_INSTRUCTION 0x08175005 //jump to $005d4014

namespace fs = boost::filesystem;

FileAddress getFileLength(const std::string& path)
{
	std::ifstream in(path, std::ios::in | std::ios::binary);
	in.seekg(0, std::ios::end);
	FileAddress length = (FileAddress)in.tellg();
	in.close();
	return length;
}

int main(int argc, char** argv)
{
	std::cout << "-Persona 3 BGM Patcher Tool, written by Aloshi-\n";
	std::cout << "http://www.aloshi.com\n\n";

	std::string mode;

	if(argc >= 2)
	{
		mode = argv[1];
	}else{
		std::cout << "USAGE: \n";
		std::cout << "  build          - compose patch data and BGM.CVM directory (./out/)\n";
		std::cout << "            (you must create BGM.CVM through an ISO builder + cvmtool's mkcvm)\n";
		std::cout << "  buildfiletable - compose remaining patch data from your final BGM.CVM\n";
		std::cout << "  patch          - apply patch to source.asm (output as source_final.asm)\n";
		std::cout << "  cvmdata        - print data about BGM.CVM\n";
		std::cout << " (specify either as a command-line argument, or type now)\n";

		std::cin >> mode;
	}

	//build music output directory and patch files
	if(mode == "build")
	{
		std::cout << "Export the entirety of BGM.CVM into ./original/\n";
		std::cout << "Place any songs you wish to add in ./additional/ORIGINALFILENAME/ \n    (filename doesn't matter, songs above 99 will probably break)\n";
		std::cout << "The original and additional folders will be merged in ./out/ \n    (new files will be renamed appropriately)\n";
		std::cout << "Part of the source code patch will be generated in ./patch/ \n";

		std::cout << "Enter Y to continue ";
		char cont;
		std::cin >> cont;

		std::cout << "\n";

		BGM bgm("original", "additional", "out");
		std::cout << "\n";

		bgm.print();

		if(!fs::is_directory("patch/"))
			fs::create_directory("patch/");

		bgm.writeSongNameTable("patch/songnametable.asm");
		bgm.writeTrackLookupTable("patch/tracklookuptable.asm");
		bgm.buildDirectory();

		std::cout << "Build complete. Use the ./out/ directory to make BGM.CVM next (create ISO, then use makecvm).\n";

	}else if(mode == "patch") //perform patch
	{
		if(!fs::is_regular_file("source.asm"))
		{
			std::cout << "Missing source.asm to patch!\n";
			return 1;
		}
		if(!fs::is_regular_file("patch/songnametable.asm") || !fs::is_regular_file("patch/tracklookuptable.asm") || !fs::is_regular_file("patch/filetable.asm"))
		{
			std::cout << "Missing patch data!  Make sure you run \"build\" and \"buildfiletable\" first!\n";
			return 1;
		}

		FileAddress origLength = getFileLength("source.asm");

		//first pass (add new data)
		Patcher p("source.asm", "source_appending.asm");

		//patch the game to jump into our routine during music changes
		//p.queueWrite(0x0010921c - PS2OFFSET, ROUTINE_JUMP_INSTRUCTION);

		//(we move some stuff around so we can use a0 and a1 in our routine)
		p.queueWrite(0x00109214 - PS2OFFSET, ROUTINE_JUMP_INSTRUCTION);
		p.queueWrite(0x00109218 - PS2OFFSET, 0x27a40050);
		p.queueWrite(0x0010921c - PS2OFFSET, 0x27858848);

		//write our music selecting routine
		FileAddress routineLen;
		BlockData* data = p.readBlockFromFile("routine.asm", &routineLen);
		if(data == NULL)
			return 1;
		p.queueWrite(ROUTINE_MEMORY_POS - PS2OFFSET, data, routineLen);
		

		//pad so we have a nice 16-byte aligned place to write
		p.queueWrite(0x008ACE86, "\0\0\0\0\0\0\0\0\0\0", 10);

		//write song name table
		FileAddress songNameTableAddr = 0x008ACE90;
		FileAddress songNameTableLen;
		data = p.readBlockFromFile("patch/songnametable.asm", &songNameTableLen);
		if(data == NULL)
			return 1;
		p.queueWrite(songNameTableAddr, data, songNameTableLen);

		//write track lookup table
		FileAddress trackLookupAddr = songNameTableAddr + songNameTableLen;
		FileAddress trackLookupLen;
		data = p.readBlockFromFile("patch/tracklookuptable.asm", &trackLookupLen);
		if(data == NULL)
			return 1;
		p.queueWrite(trackLookupAddr, data, trackLookupLen);

		//write file table
		FileAddress fileTableAddr = trackLookupAddr + trackLookupLen;
		FileAddress fileTableLen;
		data = p.readBlockFromFile("patch/filetable.asm", &fileTableLen);
		if(data == NULL)
			return 1;
		p.queueWrite(fileTableAddr, data, fileTableLen);

		p.doWrite();


			
		//second pass (update ELF header data to move heap + load our data, set final addresses)
		FileAddress newLength = getFileLength("source_appending.asm");
			
		p = Patcher("source_appending.asm", "source_final.asm");
		FileAddress lengthDiff = (newLength - origLength) + ELF_FOOTER_OFFSET;
		printf("Data appended length difference: 0x%08X\n", lengthDiff);
		p.queueIncUInt32(0x00000044, lengthDiff); //file read size

		//patch game to look up tracks from our data
		trackLookupAddr += PS2OFFSET;
		p.queueWrite(0x00109208 - PS2OFFSET, 0x3c020000 | ((trackLookupAddr & 0xffff0000) >> 16));
		p.queueWrite(0x0010920c - PS2OFFSET, 0x34420000 |  (trackLookupAddr & 0x0000ffff));

		//patch game to move the file table read so it loads from ours
		fileTableAddr += PS2OFFSET;
		std::cout << "fileTableAddr: " << std::hex << fileTableAddr << "\n";
		p.queueWrite(0x001981b0 - PS2OFFSET, 0x3c050000 | ((fileTableAddr & 0xffff0000) >> 16));
		p.queueWrite(0x001981b4 - PS2OFFSET, 0x34a50000 |  (fileTableAddr & 0x0000ffff));

		p.queueIncUInt32(0x0067F514, lengthDiff); //heap start address
		p.doWrite();

		std::cout << "Done. source_final.asm is your new SLUS file.\n";

	}else if(mode == "cvmdata")
	{
		iso_contents contents;
		if(cvm_contents("BGM.CVM", false, contents))
			return 1;

		for(unsigned int i = 0; i < contents.getNumEntries(); i++)
		{
			iso_entry e = contents.entries.at(i);
			printf("%*s - size: 0x%X, extAttrLen: 0x%X, flags: 0x%02X, extent: 0x%X\n", e.nameLen, e.name, e.size, e.extattrlen, e.flags, e.extent);
		}
	}else if(mode == "buildfiletable")
	{
		FileTable filetable("BGM.CVM");
		filetable.write("patch/filetable.asm");
	}

	return 0;
}
