#ifndef _UTIL_H_
#define _UTIL_H_

#include <string>
#include <vector>
#include <boost/filesystem.hpp>

namespace fs = boost::filesystem;


//vocabulary:
//track = logical song as understood by P3FES (e.g. "26.ADX")
//song = individual song as understood by a human (e.g. "26a.ADX")

#define TRACK_DATA_SIZE 0x000C

class Track
{
public:
	Track(fs::path origPath);

	unsigned char getSongCount() { return (unsigned char)mSongs.size(); };

	fs::path getOrigPath() { return mSongs.at(0); }

	//"resolved" filename
	std::string getSongFilename(unsigned char songIndex)
	{
		if(songIndex == 0)
			return getOrigPath().filename().string();

		std::string letter;
		letter = ('A' + songIndex - 1);

		std::string nameStr = getOrigPath().stem().string() + letter + getOrigPath().extension().string();

		return nameStr;
	}

	fs::path getSongPath(unsigned char songIndex) { return mSongs.at((unsigned int)songIndex); }

	void addSong(fs::path path) { mSongs.push_back(path); };

	unsigned int getSongDataSize()
	{
		unsigned int size = getSongFilename(0).length() + 1 + 1; //+1 for 'A', and +1 for null terminator
		while(size % 4 != 0)
			size++;
		return size;
	}

private:
	std::vector<fs::path> mSongs;
};

class BGM
{
public:
	BGM(std::string origPath, std::string additionalPath, std::string outPath);

	void writeTrackLookupTable(const std::string& outPath);
	void writeSongNameTable(const std::string& outPath);

	void buildDirectory();

	void print();
private:
	void load();
	
	unsigned int getPathAddress(unsigned int trackIndex, unsigned int offset);

	unsigned int getPatchOffset();

	Track* getTrackById(unsigned int trackId);
	bool isTrack(std::string filename);
	
	char* packSongFilename(Track* track, unsigned char id, unsigned int* lengthOut);
	//char* packSongFilename(std::string filename, unsigned int* lengthOut);

	std::string mOrigPath;
	std::string mAddPath;
	std::string mOutPath;

	std::vector<Track> mTracks;
	std::vector<fs::path> mOtherFiles;
};

#endif
