import 'semantic-ui-css/semantic.min.css'
import React, { Component } from 'react'
import { t } from 'ttag'
import * as beerCalc from './lib/beercalc'
import * as utils from './lib/utils'
import xmlExport from './lib/xmlExport'
import mdExport from './lib/mdExport'
import htmlExport from './lib/htmlExport'
import * as recipeTemplate from './lib/recipeTemplate'
import './App.css'
import Home from './Home2'
import Bar from './Bar'

import ModalPreferences from './ModalPreferences'

import Ingredients from './Ingredients'
import Profiles from './Profiles'
import FermentablePicker from './FermentablePicker'
import HopPicker from './HopPicker'
import MiscPicker from './MiscPicker'
import YeastPicker from './YeastPicker'

import RecipeMain from './RecipeMain'

class App extends Component {
  constructor (props) {
    super(props)
    this.state = {
      history: [],
      recipes: props.recipes,
      sortOption: props.sortOption,
      trash: props.trash,
      archives: props.archives,
      config: props.config,
      mash_profiles: props.mash_profiles,
      ingredients: props.ingredients,
      showPref: false,
      showBIAB: false,
      showHome: true,
      showFermentablePanel: false,
      showHopPanel: false,
      showMiscPanel: false,
      showYeastPanel: false,
      showMashPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastpanel: false,
      showIngredients: false,
      showProfiles: false,
      scale: false,
      currentRecipe: {
        style: {},
        fermentables: [],
        hops: [],
        miscs: [],
        yeasts: [],
        notes: '',
        mash: {
          mash_steps: []
        }
      },
      biab: {
        targetTemp: 67,
        strikeVol: 0,
        temp: 0,
        ratio: 0,
        mashVolume: 0
      },
      classic: {
        baseRatio: 3,
        steps: [{}],
        spargeVol: 0,
        mashVolume: 0,
        mashVolumeLastStep: 0
      },
      selectedIngredient: {},
      selectedIngredientType: '',
      selectedIngredientEditor: {},
      selectedProfileEditor: {
        mash_steps: []
      },
      selectedStepEditor: {}
    }
  }

  handleRecipeClick (item) {
    navigate(`mainRecipeView`)
    const recipe = utils.calcAll(item, this.state.config)
    if (recipe.mash.mash_steps.length === 0) {
      recipe.mash = recipeTemplate.template().mash
    }
    recipe.fermentables.map(fermentable => {
      fermentable.amountRatio = beerCalc.ingRatio(recipe.fermentables, fermentable.amount)
      return fermentable
    })

    const history = []
    recipe.historyMessage = t`Restore original`
    const classic = (recipe.mash.mash_steps.length > 0) ? utils.classic(recipe, this.state.config) : {}
    const currentRecipe = utils.sortRecipe(recipe)
    this.setState({
      history: history.concat([recipe]),
      currentRecipe,
      biab: utils.biab(recipe, this.state.config, this.state.biab.targetTemp),
      classic,
      showHome: false
    })
  }

  handleToggleBrewday () {
    const toggledState = !this.state.showBIAB
    this.setState({
      showBIAB: toggledState
    })
  }

  handleToggleHome () {
    navigate('/')
    const toggledState = !this.state.showHome
    const currentRecipe = { ...this.state.currentRecipe }

    this.setState({
      currentRecipe,
      recipes: utils.updateList(this.state.recipes, currentRecipe),
      showHome: toggledState,
      showIngredients: false,
      showProfiles: false,
      showFermentablePanel: false,
      showHopPanel: false,
      showMiscPanel: false,
      showYeastPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastPanel: false,
      classic: {
        baseRatio: 3,
        steps: [{}],
        spargeVol: 0,
        mashVolume: 0,
        mashVolumeLastStep: 0
      }
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleToggleIngredients () {
    const ingredients = { ...this.state.ingredients }
    ingredients.fermentables = utils.sortListByName(ingredients.fermentables)
    ingredients.hops = utils.sortListByName(ingredients.hops)
    ingredients.miscs = utils.sortListByName(ingredients.miscs)
    ingredients.yeasts = utils.sortListByName(ingredients.yeasts)
    this.setState({
      ingredients,
      showIngredients: !this.state.showIngredients,
      showHome: !this.state.showHome
    })
  }

  handleToggleProfiles () {
    this.setState({
      showProfiles: !this.state.showProfiles,
      showHome: !this.state.showHome
    })
  }

  handleToggleSortToName () {
    this.setState({
      sortOption: 'name'
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleToggleSortToBrewer () {
    this.setState({
      sortOption: 'brewer'
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleToggleScale () {
    this.setState({
      scale: !this.state.scale
    })
  }

  handleToggleFermentablePanel () {
    if (this.state.showFermentablePanel) {
      this.setState({ showPickFermentablePanel: false, currentRecipe: utils.sortRecipe(this.state.currentRecipe) })
    }
    this.setState({
      showFermentablePanel: !this.state.showFermentablePanel
    })
  }

  handleToggleHopPanel () {
    if (this.state.showHopPanel) {
      this.setState({ showPickHopPanel: false, currentRecipe: utils.sortRecipe(this.state.currentRecipe) })
    }
    this.setState({
      showHopPanel: !this.state.showHopPanel
    })
  }

  handleToggleMiscPanel () {
    if (this.state.showMiscPanel) {
      this.setState({ showPickMiscPanel: false, currentRecipe: utils.sortRecipe(this.state.currentRecipe) })
    }
    this.setState({
      showMiscPanel: !this.state.showMiscPanel
    })
  }

  handleToggleYeastPanel () {
    if (this.state.showYeastPanel) {
      this.setState({ showPickYeastPanel: false, currentRecipe: utils.sortRecipe(this.state.currentRecipe) })
    }
    this.setState({
      showYeastPanel: !this.state.showYeastPanel
    })
  }

  handleToggleMashPanel () {
    this.setState({ showMashPanel: !this.state.showMashPanel })
  }

  handleCloseFermentablePickerPanel () {
    this.setState({ showPickFermentablePanel: false })
  }

  handleCloseHopPickerPanel () {
    this.setState({ showPickHopPanel: false })
  }

  handleCloseMiscPickerPanel () {
    this.setState({ showPickMiscPanel: false })
  }

  handleCloseYeastPickerPanel () {
    this.setState({ showPickYeastPanel: false })
  }

  handleBiabTargetChange (e) {
    const biab = { ...this.state.biab }
    biab.targetTemp = parseFloat(e.target.value)
    const calculatedBiab = beerCalc.biabCalc(
      this.state.currentRecipe.preboilVolume,
      this.state.currentRecipe.grainWeight,
      this.state.config.General,
      biab.targetTemp
    )
    this.setState({
      biab: calculatedBiab
    })
  }

  handleClassicTempChange (index, e) {
    const newClassic = { ...this.state.classic }
    newClassic.steps[index].waterTemp = parseFloat(e.target.value)
    this.setState({
      classic: newClassic
    })

    const calculatedClassic = {}
    calculatedClassic.baseRatio = this.state.classic.baseRatio
    calculatedClassic.steps = beerCalc.stepsCalc(this.state.classic, this.state.currentRecipe.mash.mash_steps, this.state.currentRecipe.grainWeight, this.state.config.General)
    calculatedClassic.stepsVol = beerCalc.stepsVol(calculatedClassic.steps)
    calculatedClassic.spargeVol = beerCalc.processSparge(calculatedClassic.steps, this.state.config.General, this.state.currentRecipe.grainWeight, this.state.currentRecipe.preboilVolume)
    calculatedClassic.mashVolume = beerCalc.mashVolumeCalc(this.state.currentRecipe.grainWeight, calculatedClassic.steps[0].waterVol)
    calculatedClassic.mashVolumeLastStep = (calculatedClassic.mashVolume + calculatedClassic.stepsVol) - calculatedClassic.steps[0].waterVol

    console.log(this.state.classic.baseRatio)

    this.setState({
      classic: calculatedClassic
    })
  }

  handleClassicBaseRatioChange (e) {
    const newClassic = { ...this.state.classic }
    newClassic.baseRatio = parseFloat(e.target.value)
    this.setState({
      classic: newClassic
    })
    newClassic.steps = beerCalc.stepsCalc(newClassic, this.state.currentRecipe.mash.mash_steps, this.state.currentRecipe.grainWeight, this.state.config.General)
    newClassic.stepsVol = beerCalc.stepsVol(newClassic.steps)
    newClassic.spargeVol = beerCalc.processSparge(newClassic.steps, this.state.config.General, this.state.currentRecipe.grainWeight, this.state.currentRecipe.preboilVolume)
    newClassic.mashVolume = beerCalc.mashVolumeCalc(this.state.currentRecipe.grainWeight, newClassic.steps[0].waterVol)
    newClassic.mashVolumeLastStep = (newClassic.mashVolume + newClassic.stepsVol) - newClassic.steps[0].waterVol

    this.setState({
      classic: newClassic
    })
  }

  handleShowPref () {
    this.setState({ showPref: true })
  }

  handleClosePref () {
    this.setState({ showPref: false })
  }

  handleConfigGeneralChange (message, e) {
    const config = { ...this.state.config }
    config.General[message] = e.target.value
    this.setState({
      config
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleRecipeUpdate (message, e) {
    const newRecipe = { ...this.state.currentRecipe, style: { ...this.state.currentRecipe.style } }
    let ratio = 1
    if (message === 'Name updated') {
      newRecipe.name = e.target.value
    } else if (message === 'Brewer updated') {
      newRecipe.brewer = e.target.value
    } else if (message === 'Style updated') {
      newRecipe.style.name = e.target.value
    } else if (message === 'Efficiency updated') {
      newRecipe.efficiency = e.target.value
    } else if (message === 'Batch size updated') {
      newRecipe.batch_size = e.target.value
      ratio = newRecipe.batch_size / this.state.currentRecipe.batch_size
    } else if (message === 'Boil time updated') {
      newRecipe.boil_time = e.target.value
    }
    newRecipe.historyMessage = message
    const calculatedRecipe =
      (this.state.scale === true)
        ? utils.scaleAndCalc(newRecipe, ratio, this.state.config)
        : utils.calcAll(newRecipe, this.state.config)
    this.setState({
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleNotesChanged (e) {
    const currentRecipe = { ...this.state.currentRecipe }
    currentRecipe.notes = e.target.value
    currentRecipe.historyMessage = 'Notes changed'
    const recipes = utils.updateList(this.state.recipes, currentRecipe)
    const history = utils.historyAppend(this.state.history, currentRecipe)
    this.setState({ currentRecipe, recipes, history }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleRestoreRecipe (recipe, e) {
    console.log(recipe)
    const classic = (recipe.mash.mash_steps.length > 0) ? utils.classic(recipe, this.state.config) : {}
    this.setState({
      currentRecipe: recipe,
      recipes: utils.updateList(this.state.recipes, recipe),
      biab: utils.biab(recipe, this.state.config, this.state.biab.targetTemp),
      classic,
      showFermentablePanel: false,
      showHopPanel: false,
      showMiscPanel: false,
      showYeastPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleFermentableClick (item) {
    this.setState({
      showFermentablePanel: true,
      selectedIngredient: item,
      showHopPanel: false,
      showMiscPanel: false,
      showYeastPanel: false,
      showMashPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastpanel: false
    })
  }

  handleHopClick (item) {
    this.setState({
      showHopPanel: true,
      selectedIngredient: item,
      showFermentablePanel: false,
      showMiscPanel: false,
      showYeastPanel: false,
      showMashPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastpanel: false
    })
  }

  handleMiscClick (item) {
    this.setState({
      showMiscPanel: true,
      selectedIngredient: item,
      showFermentablePanel: false,
      showHopPanel: false,
      showYeastPanel: false,
      showMashPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastpanel: false
    })
  }

  handleYeastClick (item) {
    this.setState({
      showYeastPanel: true,
      selectedIngredient: item,
      showFermentablePanel: false,
      showHopPanel: false,
      showMiscPanel: false,
      showMashPanel: false,
      showPickFermentablePanel: false,
      showPickHopPanel: false,
      showPickMiscPanel: false,
      showPickYeastpanel: false
    })
  }

  handleMashDetailsClick () {
    this.setState({
      showYeastPanel: false,
      showFermentablePanel: false,
      showHopPanel: false,
      showMiscPanel: false,
      showMashPanel: true
    })
  }

  handleAddFermentableToRecipe () {
    const fermentable = { name: 'grain', id: nanoid(), amount: 0, yield: 80, color: 3, type: 'Grain', use: 'add_to_mash', amountRatio: 0 }
    const newRecipe = { ...this.state.currentRecipe, fermentables: [...this.state.currentRecipe.fermentables] }
    newRecipe.fermentables = newRecipe.fermentables.concat([fermentable])
    this.setState({
      selectedIngredient: fermentable,
      showFermentablePanel: true,
      showPickFermentablePanel: true,
      currentRecipe: newRecipe
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleAddHopToRecipe () {
    const hop = { name: 'hop', id: nanoid(), amount: 0, alpha: 0, time: 0, form: 'Leaf', use: 'Boil', ibuPart: 0, slotA: 0, slotB: 0 }
    const newRecipe = { ...this.state.currentRecipe, hops: [...this.state.currentRecipe.hops] }
    newRecipe.hops = newRecipe.hops.concat([hop])
    this.setState({
      selectedIngredient: hop,
      showHopPanel: true,
      showPickHopPanel: true,
      currentRecipe: newRecipe
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleAddMiscToRecipe () {
    const misc = { name: 'misc', id: nanoid(), amount: 0, time: 0, type: 'Spice', use: 'Boil' }
    const newRecipe = { ...this.state.currentRecipe, miscs: [...this.state.currentRecipe.miscs] }
    newRecipe.miscs = newRecipe.miscs.concat([misc])
    this.setState({
      selectedIngredient: misc,
      showMiscPanel: true,
      showPickMiscPanel: true,
      currentRecipe: newRecipe
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleAddYeastToRecipe () {
    const yeast = { name: 'yeast', id: nanoid(), amount: 0, laboratory: 'generic', product_id: 'none', attenuation: 0, form: 'Liquid' }
    const newRecipe = { ...this.state.currentRecipe, yeasts: [...this.state.currentRecipe.yeasts] }
    newRecipe.yeasts = newRecipe.yeasts.concat([yeast])
    this.setState({
      selectedIngredient: yeast,
      showYeastPanel: true,
      showPickYeastPanel: true,
      currentRecipe: newRecipe
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleFermentableChanged (message, e, data) {
    const newFermentable = { ...this.state.selectedIngredient }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.historyMessage = 'Fermentable changed'
    if (message === 'Name updated') {
      newFermentable.name = e.target.value
    } else if (message === 'Amount updated') {
      newFermentable.amount = parseFloat(e.target.value)
    } else if (message === 'Yield updated') {
      newFermentable.yield = parseFloat(e.target.value)
    } else if (message === 'Color updated') {
      newFermentable.color = parseFloat(e.target.value)
    } else if (message === 'Type updated') {
      newFermentable.type = data.value
    } else if (message === 'Use updated') {
      newFermentable.use = data.value
    }
    newRecipe.fermentables = utils.updateList(newRecipe.fermentables, newFermentable)
    newRecipe.fermentables.map(fermentable => {
      fermentable.amountRatio = beerCalc.ingRatio(newRecipe.fermentables, fermentable.amount)
      return fermentable
    })
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    this.setState({
      selectedIngredient: newFermentable,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      history: utils.historyAppend(this.state.history, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleHopChanged (message, e, data) {
    const newHop = { ...this.state.selectedIngredient }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.historyMessage = 'Hop changed'
    if (message === 'Name updated') {
      newHop.name = e.target.value
    } else if (message === 'Amount updated') {
      newHop.amount = parseFloat(e.target.value)
    } else if (message === 'Time updated') {
      newHop.time = parseFloat(e.target.value)
    } else if (message === 'Alpha updated') {
      newHop.alpha = parseFloat(e.target.value)
    } else if (message === 'Form updated') {
      newHop.form = data.value
    } else if (message === 'Use updated') {
      newHop.use = data.value
    } else if (message === 'SlotA updated') {
      newHop.slotA = data.value
    } else if (message === 'SlotB updated') {
      newHop.slotB = data.value
    }
    newRecipe.hops = utils.updateList(newRecipe.hops, newHop)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    this.setState({
      selectedIngredient: newHop,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      history: utils.historyAppend(this.state.history, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleMiscChanged (message, e, data) {
    const newMisc = { ...this.state.selectedIngredient }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.historyMessage = 'Misc changed'
    if (message === 'Name updated') {
      newMisc.name = e.target.value
    } else if (message === 'Amount updated') {
      newMisc.amount = parseFloat(e.target.value)
    } else if (message === 'Time updated') {
      newMisc.time = parseFloat(e.target.value)
    } else if (message === 'Type updated') {
      newMisc.type = data.value
    } else if (message === 'Use updated') {
      newMisc.use = data.value
    }
    newRecipe.miscs = utils.updateList(newRecipe.miscs, newMisc)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    this.setState({
      selectedIngredient: newMisc,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      history: utils.historyAppend(this.state.history, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleYeastChanged (message, e, data) {
    const newYeast = { ...this.state.selectedIngredient }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.historyMessage = 'Yeast changed'
    if (message === 'Name updated') {
      newYeast.name = e.target.value
    } else if (message === 'Lab updated') {
      newYeast.laboratory = e.target.value
    } else if (message === 'Id updated') {
      newYeast.product_id = e.target.value
    } else if (message === 'Attenuation updated') {
      newYeast.attenuation = parseFloat(e.target.value)
    } else if (message === 'Form updated') {
      newYeast.form = data.value
    }
    newRecipe.yeasts = utils.updateList(newRecipe.yeasts, newYeast)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    this.setState({
      selectedIngredient: newYeast,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      history: utils.historyAppend(this.state.history, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteIngredientRecipe (message, e) {
    const newRecipe = {
      ...this.state.currentRecipe,
      fermentables: [...this.state.currentRecipe.fermentables],
      hops: [...this.state.currentRecipe.hops],
      miscs: [...this.state.currentRecipe.miscs],
      yeasts: [...this.state.currentRecipe.yeasts]
    }
    if (message === 'fermentable') {
      newRecipe.fermentables = utils.deleteItem(newRecipe.fermentables, this.state.selectedIngredient)
      this.setState({ showFermentablePanel: false, showPickFermentablePanel: false })
    } else if (message === 'hop') {
      newRecipe.hops = utils.deleteItem(newRecipe.hops, this.state.selectedIngredient)
      this.setState({ showHopPanel: false, showPickHopPanel: false })
    } else if (message === 'misc') {
      newRecipe.miscs = utils.deleteItem(newRecipe.miscs, this.state.selectedIngredient)
      this.setState({ showMiscPanel: false, showPickMiscPanel: false })
    } else if (message === 'yeast') {
      newRecipe.yeasts = utils.deleteItem(newRecipe.yeasts, this.state.selectedIngredient)
      this.setState({ showYeastPanel: false, showPickYeastPanel: false })
    }
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    calculatedRecipe.historyMessage = `${this.state.selectedIngredient.name} deleted`
    const recipes = utils.updateList(this.state.recipes, calculatedRecipe)
    this.setState({
      recipes,
      currentRecipe: calculatedRecipe,
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handlePickFermentableClicked () {
    this.setState({ showPickFermentablePanel: true })
  }

  handlePickHopClicked () {
    this.setState({ showPickHopPanel: true })
  }

  handlePickMiscClicked () {
    this.setState({ showPickMiscPanel: true })
  }

  handlePickYeastClicked () {
    this.setState({ showPickYeastPanel: true })
  }

  handleFermentablePickerSelected (item) {
    const fermentable = {
      id: this.state.selectedIngredient.id,
      name: item.name,
      amount: this.state.selectedIngredient.amount,
      yield: item.yield,
      color: item.color,
      type: item.type,
      use: 'add_to_mash'
    }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.fermentables = utils.updateList(newRecipe.fermentables, fermentable)
    newRecipe.fermentables.map(fermentable => {
      fermentable.amountRatio = beerCalc.ingRatio(newRecipe.fermentables, fermentable.amount)
      return fermentable
    })
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    calculatedRecipe.historyMessage = 'Fermentable changed'
    this.setState({
      selectedIngredient: fermentable,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleHopPickerSelected (item) {
    const hop = {
      id: this.state.selectedIngredient.id,
      name: item.name,
      amount: this.state.selectedIngredient.amount,
      alpha: item.alpha,
      time: this.state.selectedIngredient.time,
      form: item.form,
      use: 'Boil',
      slotA: 0,
      slotB: 0
    }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.hops = utils.updateList(newRecipe.hops, hop)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    calculatedRecipe.historyMessage = 'Hop changed'
    this.setState({
      selectedIngredient: hop,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleMiscPickerSelected (item) {
    const misc = {
      id: this.state.selectedIngredient.id,
      name: item.name,
      amount: this.state.selectedIngredient.amount,
      time: this.state.selectedIngredient.time,
      type: item.type,
      use: 'Boil'
    }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.miscs = utils.updateList(newRecipe.miscs, misc)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    calculatedRecipe.historyMessage = 'Misc changed'
    this.setState({
      selectedIngredient: misc,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleYeastPickerSelected (item) {
    const yeast = {
      id: this.state.selectedIngredient.id,
      name: item.name,
      laboratory: item.laboratory,
      product_id: item.product_id,
      attenuation: item.attenuation,
      form: item.form
    }
    const newRecipe = { ...this.state.currentRecipe }
    newRecipe.yeasts = utils.updateList(newRecipe.yeasts, yeast)
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    calculatedRecipe.historyMessage = 'Yeast changed'
    this.setState({
      selectedIngredient: yeast,
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config),
      history: utils.historyAppend(this.state.history, calculatedRecipe)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleProfileChange (message, e, data) {
    const newProfile = this.state.mash_profiles.find(item => item.id === data.value)
    const newRecipe = { ...this.state.currentRecipe, mash: { ...this.state.currentRecipe.mash } }
    newRecipe.historyMessage = 'Profile changed'
    newRecipe.mash = newProfile
    const calculatedRecipe = utils.calcAll(newRecipe, this.state.config)
    this.setState({
      currentRecipe: calculatedRecipe,
      recipes: utils.updateList(this.state.recipes, calculatedRecipe),
      history: utils.historyAppend(this.state.history, calculatedRecipe),
      biab: utils.biab(calculatedRecipe, this.state.config, this.state.biab.targetTemp),
      classic: utils.classic(calculatedRecipe, this.state.config)
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleIngredientTypeChanged (message, e) {
    const empty = { name: '', color: '', alpha: '' }
    this.setState({ selectedIngredientEditor: empty })
    this.setState({ selectedIngredientType: message })
  }

  handleIngredientEditorSelected (ingredient, e) {
    this.setState({
      selectedIngredientEditor: ingredient
    })
  }

  handleFermentableEditorChanged (message, e, data) {
    const newIngredient = { ...this.state.selectedIngredientEditor }
    const newIngredientsList = { ...this.state.ingredients, fermentables: [...this.state.ingredients.fermentables] }
    if (message === 'Name updated') {
      newIngredient.name = e.target.value
    } else if (message === 'Yield updated') {
      newIngredient.yield = parseFloat(e.target.value)
    } else if (message === 'Color updated') {
      newIngredient.color = parseFloat(e.target.value)
    } else if (message === 'Type updated') {
      newIngredient.type = data.value
    }
    newIngredientsList.fermentables = utils.updateList(newIngredientsList.fermentables, newIngredient)
    this.setState({
      selectedIngredientEditor: newIngredient,
      ingredients: newIngredientsList
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleHopEditorChanged (message, e, data) {
    const newIngredient = { ...this.state.selectedIngredientEditor }
    const newIngredientsList = { ...this.state.ingredients, hops: [...this.state.ingredients.hops] }
    if (message === 'Name updated') {
      newIngredient.name = e.target.value
    } else if (message === 'Form updated') {
      newIngredient.form = data.value
    } else if (message === 'Alpha updated') {
      newIngredient.alpha = e.target.value
    }
    newIngredientsList.hops = utils.updateList(newIngredientsList.hops, newIngredient)
    this.setState({
      selectedIngredientEditor: newIngredient,
      ingredients: newIngredientsList
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleMiscEditorChanged (message, e, data) {
    const newIngredient = { ...this.state.selectedIngredientEditor }
    const newIngredientsList = { ...this.state.ingredients, miscs: [...this.state.ingredients.miscs] }
    if (message === 'Name updated') {
      newIngredient.name = e.target.value
    } else if (message === 'Type updated') {
      newIngredient.type = data.value
    }
    newIngredientsList.miscs = utils.updateList(newIngredientsList.miscs, newIngredient)
    this.setState({
      selectedIngredientEditor: newIngredient,
      ingredients: newIngredientsList
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleYeastEditorChanged (message, e, data) {
    const newIngredient = { ...this.state.selectedIngredientEditor }
    const newIngredientsList = { ...this.state.ingredients, yeasts: [...this.state.ingredients.yeasts] }
    if (message === 'Name updated') {
      newIngredient.name = e.target.value
    } else if (message === 'Form updated') {
      newIngredient.form = data.value
    } else if (message === 'Laboratory updated') {
      newIngredient.laboratory = e.target.value
    } else if (message === 'Product_id updated') {
      newIngredient.product_id = e.target.value
    } else if (message === 'Attenuation updated') {
      newIngredient.attenuation = e.target.value
    }
    newIngredientsList.yeasts = utils.updateList(newIngredientsList.yeasts, newIngredient)
    this.setState({
      selectedIngredientEditor: newIngredient,
      ingredients: newIngredientsList
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleAddIngredient (item, type, e) {
    const newItem = { ...item }
    const newIngredientsList = {
      ...this.state.ingredients,
      fermentables: [...this.state.ingredients.fermentables],
      hops: [...this.state.ingredients.hops],
      miscs: [...this.state.ingredients.miscs],
      yeasts: [...this.state.ingredients.yeasts]
    }
    newItem.id = nanoid()
    if (type === 'fermentable') {
      newIngredientsList.fermentables.unshift(newItem)
    } else if (type === 'hop') {
      newIngredientsList.hops.unshift(newItem)
    } else if (type === 'misc') {
      newIngredientsList.miscs.unshift(newItem)
    } else if (type === 'yeast') {
      newIngredientsList.yeasts.unshift(newItem)
    }
    this.setState({
      selectedIngredientEditor: newItem,
      ingredients: newIngredientsList
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteIngredientEditor (message, e) {
    const empty = { name: '', color: '', alpha: '' }
    const newIngredientsList = {
      ...this.state.ingredients,
      fermentables: [...this.state.ingredients.fermentables],
      hops: [...this.state.ingredients.hops],
      miscs: [...this.state.ingredients.miscs],
      yeasts: [...this.state.ingredients.yeasts]
    }
    if (message === 'fermentables') {
      newIngredientsList.fermentables =
        utils.deleteItem(newIngredientsList.fermentables, this.state.selectedIngredientEditor)
    } else if (message === 'hops') {
      newIngredientsList.hops =
        utils.deleteItem(newIngredientsList.hops, this.state.selectedIngredientEditor)
    } else if (message === 'miscs') {
      newIngredientsList.miscs =
        utils.deleteItem(newIngredientsList.miscs, this.state.selectedIngredientEditor)
    } else if (message === 'yeasts') {
      newIngredientsList.yeasts =
        utils.deleteItem(newIngredientsList.yeasts, this.state.selectedIngredientEditor)
    }
    this.setState({
      ingredients: newIngredientsList,
      selectedIngredientEditor: empty
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleProfileSelectedEditor (item, e) {
    const empty = {
      name: '', step_temp: '', step_time: '', type: ''
    }
    this.setState({
      selectedProfileEditor: item,
      selectedStepEditor: empty
    })
  }

  handleStepSelectedEditor (item, e) {
    this.setState({
      selectedStepEditor: item
    })
  }

  handleProfileBaseEditorChanged (message, e) {
    const newProfile = { ...this.state.selectedProfileEditor }
    if (message === 'Name updated') {
      newProfile.name = e.target.value
    } else if (message === 'Ph updated') {
      newProfile.ph = parseFloat(e.target.value)
    } else if (message === 'Sparge updated') {
      newProfile.sparge_temp = parseFloat(e.target.value)
    }
    const newProfiles = utils.updateList(this.state.mash_profiles, newProfile)
    this.setState({
      mash_profiles: newProfiles,
      selectedProfileEditor: newProfile
    })
  }

  handleStepEditorChanged (message, e, data) {
    const newProfile = {
      ...this.state.selectedProfileEditor,
      mash_steps: [this.state.selectedProfileEditor.mash_steps]
    }
    const newStep = { ...this.state.selectedStepEditor }

    if (message === 'Name updated') {
      newStep.name = e.target.value
    } else if (message === 'Temp updated') {
      newStep.step_temp = parseFloat(e.target.value)
    } else if (message === 'Time updated') {
      newStep.step_time = parseFloat(e.target.value)
    } else if (message === 'Type updated') {
      newStep.type = data.value
    }
    newProfile.mash_steps = utils.updateList(this.state.selectedProfileEditor.mash_steps, newStep)
    const newProfiles = utils.updateList(this.state.mash_profiles, newProfile)
    this.setState({
      selectedStepEditor: newStep,
      selectedProfileEditor: newProfile,
      mash_profiles: newProfiles
    })
  }

  handleAddNewProfile () {
    const newProfile = {
      name: 'New Profile',
      ph: 5.4,
      sparge_temp: 78,
      mash_steps: [],
      id: nanoid()
    }
    const profiles = [...this.state.mash_profiles]
    profiles.push(newProfile)
    this.setState({
      mash_profiles: profiles
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleAddNewStep () {
    const newStep = {
      name: 'New Step',
      type: 'Infusion',
      step_time: 60,
      step_temp: 65,
      id: nanoid()
    }
    const profile = { ...this.state.selectedProfileEditor, mash_steps: [...this.state.selectedProfileEditor.mash_steps] }
    profile.mash_steps.push(newStep)
    const profiles = utils.updateList(this.state.mash_profiles, profile)
    this.setState({
      selectedProfileEditor: profile,
      mash_profiles: profiles
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteProfile () {
    const mashProfiles = utils.deleteItem(this.state.mash_profiles, this.state.selectedProfileEditor)
    const emptyStep = {
      name: '', step_temp: '', step_time: '', type: ''
    }
    const emptyProfile = {
      name: '', ph: '', sparge_temp: '', mash_steps: []
    }
    this.setState({
      mash_profiles: mashProfiles,
      selectedProfileEditor: emptyProfile,
      selectedStepEditor: emptyStep
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteStep () {
    const profile = { ...this.state.selectedProfileEditor, mash_steps: [...this.state.selectedProfileEditor.mash_steps] }
    profile.mash_steps = utils.deleteItem(profile.mash_steps, this.state.selectedStepEditor)
    const mashProfiles = utils.updateList(this.state.mash_profiles, profile)
    const emptyStep = {
      name: '', step_temp: '', step_time: '', type: ''
    }
    this.setState({
      mash_profiles: mashProfiles,
      selectedStepEditor: emptyStep,
      selectedProfileEditor: profile
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleFileSaveDialog (message, e) {
    const filters = () => {
      if (message === 'md') {
        return [{ name: 'markdown', extensions: ['md'] }]
      } else if (message === 'xml') {
        return [{ name: 'xml', extensions: ['xml'] }]
      } else if (message === 'html') {
        return [{ name: 'html', extensions: ['html'] }]
      } else if (message === 'jb4') {
        return [{ name: 'jb4', extensions: ['jb4'] }]
      }
      return [{ name: 'xml', extensions: ['xml'] }]
    }

    const fileContent = () => {
      if (message === 'md') {
        return mdExport(this.state.currentRecipe)
      } else if (message === 'xml') {
        return xmlExport(this.state.currentRecipe)
      } else if (message === 'html') {
        return htmlExport(this.state.currentRecipe)
      } else if (message === 'jb4') {
        const recipe = this.state.currentRecipe
        recipe.snapshots = []
        recipe.isValid = true
        return JSON.stringify(recipe)
      }
      return xmlExport(this.state.currentRecipe)
    }

    dialog.showSaveDialog({
      filters: filters()
    }, (fileName) => {
      if (fileName === undefined) return
      fs.writeFile(fileName, fileContent(), (err) => {
      })
    })
    this.setState({

    })
  }

  handleOpenLegacyDirectory () {
    ipcRenderer.send('importLegacyDirectory')
    ipcRenderer.on('recipesLegacyJson', (event, data) => {
      this.setState({
        recipes: this.state.recipes.concat(data)
      }, () => ipcRenderer.send('writeBFF', this.state))
    })
  }

  handleOpenFiles () {
    const options = {
      properties: ['openFile', 'multiSelections']
    }
    dialog.showOpenDialog(options, paths => {
      if (paths) {
        ipcRenderer.send('importFiles', paths)
      }
      ipcRenderer.once('recipesJson', (event, data) => {
        const recipesList = data
        recipesList.map(utils.cleanRecipe)

        this.setState({
          recipes: this.state.recipes.concat(recipesList)
        }, () => ipcRenderer.send('writeBFF', this.state))
      })
    })
  }

  handleImportJb4 () {
    const options = {
      properties: ['openFile']
    }
    dialog.showOpenDialog(options, path => {
      if (path) {
        const recipe = JSON.parse(fs.readFileSync(path[0], 'utf8'))
        recipe.id = nanoid()
        if (recipe.isValid) {
          const notifImport = new Notification('Joliebulle', {
            body: 'Import completed succesfully'
          })
          this.setState({
            recipes: this.state.recipes.concat(recipe)
          }, () => ipcRenderer.send('writeBFF', this.state))
        } else {
          const notifFailedImport = new Notification('Joliebulle', {
            body: 'Recipe is not valid'
          })
        }
      }
    })
  }

  handleOpenLegacyIngredients () {
    ipcRenderer.send('importIngredients')
    ipcRenderer.on('ingredientsJson', (event, data) => {
      const ingredients = { ...this.state.ingredients }
      ingredients.fermentables = ingredients.fermentables.concat(data.fermentables)
      ingredients.hops = ingredients.hops.concat(data.hops)
      ingredients.miscs = ingredients.miscs.concat(data.miscs)
      ingredients.yeasts = ingredients.yeasts.concat(data.yeasts)
      this.setState({
        ingredients
      }, () => ipcRenderer.send('writeBFF', this.state))
    })
  }

  handleRestoreIngredientsBase () {
    ipcRenderer.send('restoreDatabase')
    ipcRenderer.on('originalIngredients', (event, data) => {
      const ingredients = data
      this.setState({ ingredients }, () => ipcRenderer.send('writeBFF', this.state))
    })
  }

  handleNewRecipe () {
    const currentRecipe = recipeTemplate.template()
    const history = []
    currentRecipe.historyMessage = t`Restore original`
    const recipes = [...this.state.recipes]
    recipes.unshift(currentRecipe)
    const classic = (currentRecipe.mash.mash_steps.length > 0) ? utils.classic(currentRecipe, this.state.config) : {}
    const biab = utils.biab(currentRecipe, this.state.config, this.state.biab.targetTemp)
    navigate(`mainRecipeView`)
    this.setState({
      history: history.concat([currentRecipe]),
      currentRecipe,
      recipes,
      biab,
      classic,
      showHome: false
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteRecipe (item) {
    const recipes = utils.deleteItem(this.state.recipes, item)
    const trash = [...this.state.trash].concat([item])
    this.setState({
      recipes,
      trash
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleEmptyTrash () {
    const trash = []
    this.setState({ trash }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleTrashToInbox (item) {
    const trash = utils.deleteItem(this.state.trash, item)
    const recipes = this.state.recipes.concat([item])
    this.setState({ trash, recipes }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleMoveToArchive (item) {
    const recipes = utils.deleteItem(this.state.recipes, item)
    const archives = [...this.state.archives].concat([item])
    this.setState({ recipes, archives }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleDeleteFromArchives (item) {
    const archives = utils.deleteItem(this.state.archives, item)
    const trash = [...this.state.trash].concat([item])
    this.setState({
      archives,
      trash
    }, () => ipcRenderer.send('writeBFF', this.state))
  }

  handleArchiveToInbox (item) {
    const recipes = [...this.state.recipes].concat([item])
    const archives = utils.deleteItem(this.state.archives, item)
    this.setState({ recipes, archives }, () => ipcRenderer.send('writeBFF', this.state))
  }

  render () {
    return (
      <div className='App'>
        <Bar
          recipe={this.state.currentRecipe}
          toggleHome={() => this.handleToggleHome()}
          onShowPrefClick={() => this.handleShowPref()}
          showHome={this.state.showHome}
          showIngredients={this.state.showIngredients}
          showProfiles={this.state.showProfiles}
        />
        <LocationProvider history={history}>
          <Router>
            <Home
              path='/'
              recipes={this.state.recipes}
              sortOption={this.state.sortOption}
              onSortByName={() => this.handleToggleSortToName()}
              onSortByBrewer={() => this.handleToggleSortToBrewer()}
              trash={this.state.trash}
              archives={this.state.archives}
              onRecipeClick={item => this.handleRecipeClick(item)}
              showHome={this.state.showHome}
              searchTerm={this.state.searchTerm}
              onIngredientsToggle={() => this.handleToggleIngredients()}
              onProfilesToggle={() => this.handleToggleProfiles()}
              onImportLegacyDirectory={() => this.handleOpenLegacyDirectory()}
              onImportJb4={() => this.handleImportJb4()}
              onOpenFiles={() => this.handleOpenFiles()}
              onNewRecipe={() => this.handleNewRecipe()}
              onDeleteRecipe={item => this.handleDeleteRecipe(item)}
              onEmptyTrash={() => this.handleEmptyTrash()}
              onTrashToInbox={item => this.handleTrashToInbox(item)}
              onArchive={item => this.handleMoveToArchive(item)}
              onArchiveToInbox={item => this.handleArchiveToInbox(item)}
              onDeleteFromArchives={item => this.handleDeleteFromArchives(item)}
            />
            <RecipeMain
              path='mainRecipeView'
              showHome={this.state.showHome}
              showIngredients={this.state.showIngredients}
              showProfiles={this.state.showProfiles}
              recipe={this.state.currentRecipe}
              selectedIngredient={this.state.selectedIngredient}
              history={this.state.history}
              config={this.state.config}
              biab={this.state.biab}
              classic={this.state.classic}
              mash_profiles={this.state.mash_profiles}
              handleBiabTargetChange={e => this.handleBiabTargetChange(e)}
              handleClassicTempChange={(index, e) => this.handleClassicTempChange(index, e)}
              handleClassicBaseRatioChange={e => this.handleClassicBaseRatioChange(e)}
              showBIAB={this.state.showBIAB}
              toggleBrewday={() => this.handleToggleBrewday()}
              toggleScale={() => this.handleToggleScale()}
              onRecipeSizeChanged={e => this.handleRecipeSizeUpdated(e)}
              onRecipeEfficiencyChanged={e => this.handleRecipeEfficiencyUpdated(e)}
              onExportClicked={(message, e) => this.handleFileSaveDialog(message, e)}
              onRecipeChanged={(message, e) => this.handleRecipeUpdate(message, e)}
              onNotesChanged={e => this.handleNotesChanged(e)}
              onRestoreRecipeClicked={(recipe, e) => this.handleRestoreRecipe(recipe, e)}
              onFermentableClick={item => this.handleFermentableClick(item)}
              onFermentablePanelToggle={() => this.handleToggleFermentablePanel()}
              showFermentablePanel={this.state.showFermentablePanel}
              onHopPanelToggle={() => this.handleToggleHopPanel()}
              onHopClick={item => this.handleHopClick(item)}
              showHopPanel={this.state.showHopPanel}
              onMiscClick={item => this.handleMiscClick(item)}
              showMiscPanel={this.state.showMiscPanel}
              onMiscPanelToggle={() => this.handleToggleMiscPanel()}
              onYeastClick={item => this.handleYeastClick(item)}
              showYeastPanel={this.state.showYeastPanel}
              onYeastPanelToggle={() => this.handleToggleYeastPanel()}
              onFermentableChanged={(message, e, data) => this.handleFermentableChanged(message, e, data)}
              onHopChanged={(message, e, data) => this.handleHopChanged(message, e, data)}
              onMiscChanged={(message, e, data) => this.handleMiscChanged(message, e, data)}
              onYeastChanged={(message, e, data) => this.handleYeastChanged(message, e, data)}
              onMashDetailsClick={() => this.handleMashDetailsClick()}
              showMashPanel={this.state.showMashPanel}
              onMashPanelToggle={() => this.handleToggleMashPanel()}
              onProfileChanged={(message, e, data) => this.handleProfileChange(message, e, data)}
              deleteIngredient={(message, e) => this.handleDeleteIngredientRecipe(message, e)}
              onPickFermentableClicked={() => this.handlePickFermentableClicked()}
              onPickHopClicked={() => this.handlePickHopClicked()}
              onPickMiscClicked={() => this.handlePickMiscClicked()}
              onPickYeastClicked={() => this.handlePickYeastClicked()}
              onAddFermentableToRecipe={() => this.handleAddFermentableToRecipe()}
              onAddHopToRecipe={() => this.handleAddHopToRecipe()}
              onAddMiscToRecipe={() => this.handleAddMiscToRecipe()}
              onAddYeastToRecipe={() => this.handleAddYeastToRecipe()}
            />
          </Router>
        </LocationProvider>
        <Ingredients
          ingredients={this.state.ingredients}
          showIngredients={this.state.showIngredients}
          onIngredientsToggle={() => this.handleToggleIngredients}
          selectedIngredientType={this.state.selectedIngredientType}
          onIngredientTypeChanged={(message, e) => this.handleIngredientTypeChanged(message, e)}
          onIngredientEditorSelected={(ingredient, e) => this.handleIngredientEditorSelected(ingredient, e)}
          selectedIngredientEditor={this.state.selectedIngredientEditor}
          onFermentableEditorChanged={(message, e, data) => this.handleFermentableEditorChanged(message, e, data)}
          onHopEditorChanged={(message, e, data) => this.handleHopEditorChanged(message, e, data)}
          onMiscEditorChanged={(message, e, data) => this.handleMiscEditorChanged(message, e, data)}
          onYeastEditorChanged={(message, e, data) => this.handleYeastEditorChanged(message, e, data)}
          addIngredient={(item, type, e) => this.handleAddIngredient(item, type, e)}
          deleteIngredient={(message, e) => this.handleDeleteIngredientEditor(message, e)}
          onOpenIngredientsLegacy={() => this.handleOpenLegacyIngredients()}
          onRestoreIngredients={() => this.handleRestoreIngredientsBase()}
        />
        <Profiles
          showProfiles={this.state.showProfiles}
          profiles={this.state.mash_profiles}
          selectedProfile={this.state.selectedProfileEditor}
          selectedStep={this.state.selectedStepEditor}
          onProfileClicked={(item, e) => this.handleProfileSelectedEditor(item, e)}
          onStepClicked={(item, e) => this.handleStepSelectedEditor(item, e)}
          onProfileBaseChanged={(message, e) => this.handleProfileBaseEditorChanged(message, e)}
          onStepChanged={(message, e, data) => this.handleStepEditorChanged(message, e, data)}
          addProfile={() => this.handleAddNewProfile()}
          addStep={() => this.handleAddNewStep()}
          deleteProfile={() => this.handleDeleteProfile()}
          deleteStep={() => this.handleDeleteStep()}
        />

        <FermentablePicker
          showPanel={this.state.showPickFermentablePanel}
          onClosePanel={() => this.handleCloseFermentablePickerPanel()}
          fermentables={this.state.ingredients.fermentables}
          onFermentableSelected={item => this.handleFermentablePickerSelected(item)}
        />
        <HopPicker
          showPanel={this.state.showPickHopPanel}
          onClosePanel={() => this.handleCloseHopPickerPanel()}
          hops={this.state.ingredients.hops}
          onHopSelected={item => this.handleHopPickerSelected(item)}
        />
        <MiscPicker
          showPanel={this.state.showPickMiscPanel}
          onClosePanel={() => this.handleCloseMiscPickerPanel()}
          miscs={this.state.ingredients.miscs}
          onMiscSelected={item => this.handleMiscPickerSelected(item)}
        />
        <YeastPicker
          showPanel={this.state.showPickYeastPanel}
          onClosePanel={() => this.handleCloseYeastPickerPanel()}
          yeasts={this.state.ingredients.yeasts}
          onYeastSelected={item => this.handleYeastPickerSelected(item)}
        />
        <ModalPreferences
          show={this.state.showPref}
          onClosePrefClick={() => this.handleClosePref()}
          config={this.state.config}
          handleConfigGeneralChange={(message, e) => this.handleConfigGeneralChange(message, e)}
        />

      </div>
    )
  }
}

export default App
