import React, { useState } from 'react'
import * as utils from './lib/utils'
import DilutionTool from './Tools/DilutionTool'
import DecoctionTool from './Tools/DecoctionTool'
import HydrometerTool from './Tools/HydrometerTool'
import BoiloffTool from './Tools/BoiloffTool'
import AlcoolTool from './Tools/AlcoolTool'
import SgPlatoTool from './Tools/SgPlatoTool'
import PlatoAlcTool from './Tools/PlatoAlcTool'
import { t } from 'ttag'
import { Icon, Dropdown } from 'semantic-ui-react'
import SearchInput, { createFilter } from 'react-search-input'
import styled from 'styled-components/macro'
import { Box, Text, Flex } from 'rebass'
import Tooltip from '@reach/tooltip'
import '@reach/tooltip/styles.css'
import {
  Menu,
  MenuList,
  MenuButton,
  MenuItem
} from '@reach/menu-button'
import '@reach/menu-button/styles.css'

const RecipesList = styled.div`
  display : ${props => props.show ? 'block' : 'none'};
  margin-bottom:35px;
`

const RecipeLine = styled(Flex)`
  cursor:default;
  :hover {
    background:#F7F9FD;
    box-shadow: 3px 0px 0px 0px #1070CA inset;
  }
  `
const StyledInboxSelect = styled(Text)`
  cursor: pointer;
  color: ${props => props.inboxHighlightClass ? '#425A70' : '#7B8B9A'};
`
const StyledArchivesSelect = styled(Text)`
  cursor: pointer;
  color: ${props => props.archivesHighlightClass ? '#425A70' : '#7B8B9A'};
`
const StyledTrashSelect = styled(Text)`
  cursor: pointer;
  color: ${props => props.trashHighlightClass ? '#425A70' : '#7B8B9A'};
`
const StyledSortBox = styled(Box)`
 position:fixed;
 color:#1070CA;
 font-weight:500;
 z-index: 1000;
 top:7px;
 left:560px;
`
const NewRecipeButton = styled(Box)`
 position:fixed;
 color:#1070CA;
 font-weight:500;
 z-index: 1000;
 top:7px;
 left:650px;
 cursor: pointer;
`
const StyledStatusBar = styled.div`
  position:fixed;
  z-index:1000;
  bottom:0;
  width:100%;
  height:23px;
  color:#fff;
  background:#1070CA;
`
const StyledStatusButton = styled.button`
 border:none;
 color:#fff;
 background:transparent;
 font-size: 14px;
 margin-left:35px;
 margin-top:2px;
 cursor: pointer;
`

const StyledMenuButton = styled(MenuButton)`
  border:none;
  color:#fff;
  background:transparent;
  font-size: 14px;
  margin-left:35px;
  margin-top:2px;
  cursor: pointer;
`

const StyledMenuList = styled(MenuList)`
  background: white;
  color: #425A70;
  > [data-reach-menu-item] {
    font-size: 14px;
  }
  > [data-reach-menu-item][data-selected] {
    background: #1070CA;
    color: white;
  }
`

const StyledEmptyButton = styled.button`
 border:none;
 color:#1070CA;
 font-weight:500;
 background:transparent;
 margin-left:55px;
 margin-bottom:35px;
 cursor: pointer;
 text-decoration:underline;
`

function Home (props) {
  const showClass = props.showHome ? 'home display-block' : 'display-none'

  const [searchTerm, setSearchTerm] = useState('')

  const KEYS_TO_FILTERS = ['name']
  const filteredRecipes = utils.sortRecipesList(
    props.recipes.filter(createFilter(searchTerm, KEYS_TO_FILTERS)),
    props.sortOption)

  const filteredTrash = utils.sortRecipesList(
    props.trash.filter(createFilter(searchTerm, KEYS_TO_FILTERS)),
    props.sortOption
  )

  const filteredArchives = utils.sortRecipesList(
    props.archives.filter(createFilter(searchTerm, KEYS_TO_FILTERS)),
    props.sortOption)

  const [showRecipes, setShowRecipes] = useState(true)

  const [showTrash, setShowTrash] = useState(false)

  const [showArchives, setShowArchives] = useState(false)

  const importClass = (showTrash || showArchives) ? 'display-none' : 'import-recipe-button'

  const [dialogShowDecoction, setDialogShowDecoction] = useState(false)
  const [dialogShowDilution, setDialogShowDilution] = useState(false)
  const [dialogShowHydrometer, setDialogShowHydrometer] = useState(false)
  const [dialogShowBoiloff, setDialogShowBoiloff] = useState(false)
  const [dialogShowAlc, setDialogShowAlc] = useState(false)
  const [dialogShowSgPlato, setDialogShowSgPlato] = useState(false)
  const [dialogShowFgPlato, setDialogShowFgPlato] = useState(false)

  return (
    <div className={showClass}>
      <div className='recipes-list'>
        <div className='row-wrapper'>
          <SearchInput className='search' placeholder='Search...' onChange={setSearchTerm} />
          <Dropdown labeled floating text={t`Import`} className={importClass}>
            <Dropdown.Menu>
              <Dropdown.Item text={t`Import recipe`} onClick={props.onImportJb4} />
              <Dropdown.Item text={t`Import BeerXML file`} onClick={props.onOpenFiles} />
              <Dropdown.Item text={t`Import joliebulle 3 library`} onClick={props.onImportLegacyDirectory} />
            </Dropdown.Menu>
          </Dropdown>
        </div>
        <StyledSortBox>
          <Dropdown labeled floating text={t`Sort`} style={{ zIndex: 1500, fontWeight: 500 }}>
            <Dropdown.Menu>
              <Dropdown.Item text={t`Sort by name`} onClick={props.onSortByName} />
              <Dropdown.Item text={t`Sort by brewer`} onClick={props.onSortByBrewer} />
            </Dropdown.Menu>
          </Dropdown>
        </StyledSortBox>

        <Flex ml={5} mb={5} mt={4}>
          <StyledInboxSelect fontSize={3} inboxHighlightClass={showRecipes}
            onClick={() => { setShowTrash(false); setShowRecipes(true); setShowArchives(false) }}
          >
            <Icon className='yellow-text' name='inbox' style={{ marginRight: '10px' }} />
            Inbox
          </StyledInboxSelect>
          <Text mr={5} ml={2} pt={'2px'} fontSize={1} color='#7B8B9A'>{props.recipes.length}</Text>
          <StyledArchivesSelect fontSize={3} archivesHighlightClass={showArchives}
            onClick={() => { setShowTrash(false); setShowRecipes(false); setShowArchives(true) }}
          >
            {t`Archives`}
          </StyledArchivesSelect>
          <Text mr={5} ml={2} pt={'2px'} fontSize={1} color='#7B8B9A'>{props.archives.length}</Text>
          <StyledTrashSelect fontSize={3} trashHighlightClass={showTrash}
            onClick={() => { setShowTrash(true); setShowRecipes(false); setShowArchives(false) }}
          >
            {t`Trash`}
          </StyledTrashSelect>
        </Flex>

        <RecipesList show={showRecipes}>
          <NewRecipeButton onClick={props.onNewRecipe}>
            <Text><Icon name='plus' />{t`New Recipe`}</Text>
          </NewRecipeButton>
          {filteredRecipes.map(item => (
            <RecipeLine my={1} flexDirection='row' justifyContent='space-between' key={item.id}>
              <Box width={1} py={2} onClick={() => props.onRecipeClick(item)}>
                <Flex flexDirection='row' width={1}>
                  <Box width={1 / 4} pl={5} >
                    <Text fontWeight='600'>{item.brewer}</Text>
                  </Box>
                  <Box width={1 / 2} >
                    <Text fontWeight='500'>{item.name}</Text>
                  </Box>
                </Flex>
              </Box>
              <Flex flexDirection='row' pr={5} py={2}>
                <Tooltip
                  label={t`Archive`}
                  style={{
                    background: 'hsla(0, 0%, 0%, 0.75)',
                    color: 'white',
                    border: 'none',
                    borderRadius: '4px',
                    padding: '0.5em 1em'
                  }}
                >
                  <Box pr={4} color='#E4E7EB' fontSize={2} onClick={() => props.onArchive(item)}>
                    <Icon name='archive' />
                  </Box>
                </Tooltip>
                <Tooltip
                  label={t`Delete`}
                  style={{
                    background: 'hsla(0, 0%, 0%, 0.75)',
                    color: 'white',
                    border: 'none',
                    borderRadius: '4px',
                    padding: '0.5em 1em'
                  }}
                >
                  <Box color='#E4E7EB' fontSize={2} onClick={() => props.onDeleteRecipe(item)}><Icon name='trash alternate' /></Box>
                </Tooltip>
              </Flex>
            </RecipeLine>
          ))}
        </RecipesList>

        <RecipesList show={showTrash}>
          <StyledEmptyButton onClick={props.onEmptyTrash}>
            { t`Empty Trash` }
          </StyledEmptyButton>
          {filteredTrash.map(item => (
            <RecipeLine my={1} py={2} flexDirection='row' justifyContent='space-between' key={item.id} onClick={() => props.onTrashToInbox(item)}>
              <Box width={1}>
                <Flex flexDirection='row' width={1}>
                  <Box width={1 / 4} pl={5}>
                    <Text fontWeight='600'>{item.brewer}</Text>
                  </Box>
                  <Box width={1 / 2}>
                    <Text fontWeight='500'>{item.name}</Text>
                  </Box>
                </Flex>
              </Box>
              <Flex flexDirection='row' pr={5} width={1 / 4}>
                <Box color='#E4E7EB' fontSize={2} style={{ cursor: 'pointer' }}>{ t`Click to Restore` }</Box>
              </Flex>
            </RecipeLine>
          ))}
        </RecipesList>

        <RecipesList show={showArchives}>
          {filteredArchives.map(item => (
            <RecipeLine my={1} flexDirection='row' justifyContent='space-between' key={item.id}>
              <Box width={1} py={2}>
                <Flex flexDirection='row' width={1}>
                  <Box width={1 / 4} pl={5}>
                    <Text fontWeight='600'>{item.brewer} </Text>
                  </Box>
                  <Box width={1 / 2}>
                    <Text fontWeight='500'>{item.name}</Text>
                  </Box>
                </Flex>
              </Box>
              <Flex flexDirection='row' pr={5} py={2}>
                <Tooltip
                  label={t`Move to inbox`}
                  style={{
                    background: 'hsla(0, 0%, 0%, 0.75)',
                    color: 'white',
                    border: 'none',
                    borderRadius: '4px',
                    padding: '0.5em 1em'
                  }}
                >
                  <Box pr={4} color='#E4E7EB' fontSize={2} onClick={() => props.onArchiveToInbox(item)}><Icon name='reply' /></Box>
                </Tooltip>
                <Tooltip
                  label={t`Trash`}
                  style={{
                    background: 'hsla(0, 0%, 0%, 0.75)',
                    color: 'white',
                    border: 'none',
                    borderRadius: '4px',
                    padding: '0.5em 1em'
                  }}
                >
                  <Box color='#E4E7EB' fontSize={2} onClick={() => props.onDeleteFromArchives(item)}><Icon name='trash alternate' /></Box>
                </Tooltip>
              </Flex>

            </RecipeLine>
          ))}
        </RecipesList>

      </div>
      <StyledStatusBar>
        <StyledStatusButton onClick={props.onIngredientsToggle}>{t`Manage Ingredients`}</StyledStatusButton>
        <StyledStatusButton onClick={props.onProfilesToggle}>{t`Manage Mash Profiles`}</StyledStatusButton>
        <Menu>
          <StyledMenuButton>
            {t`Tools`} <span aria-hidden>▴</span>
          </StyledMenuButton>
          <StyledMenuList>
            <MenuItem onSelect={() => setDialogShowDecoction(true)}>{t`Decoction`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowDilution(true)}>{t`Dilution`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowHydrometer(true)}>{t`Hydrometer correction`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowBoiloff(true)}>{t`Boil Off`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowAlc(true)}>{t`Alcohol by vol`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowSgPlato(true)}>{t`Specific Gravity - Plato`}</MenuItem>
            <MenuItem onSelect={() => setDialogShowFgPlato(true)}>{t`Final Gravity - Brix (with alc)`}</MenuItem>
          </StyledMenuList>
        </Menu>
      </StyledStatusBar>

      <DilutionTool dialogShowDilution={dialogShowDilution} closeDialogDilution={() => setDialogShowDilution(false)} />
      <DecoctionTool dialogShowDecoction={dialogShowDecoction} closeDialogDecoction={() => setDialogShowDecoction(false)} />
      <HydrometerTool dialogShowHydrometer={dialogShowHydrometer} closeDialogHydrometer={() => setDialogShowHydrometer(false)} />
      <BoiloffTool dialogShowBoiloff={dialogShowBoiloff} closeDialogBoiloff={() => setDialogShowBoiloff(false)} />
      <AlcoolTool dialogShowAlc={dialogShowAlc} closeDialogAlc={() => setDialogShowAlc(false)} />
      <SgPlatoTool dialogShowSgPlato={dialogShowSgPlato} closeDialogSgPlato={() => setDialogShowSgPlato(false)} />
      <PlatoAlcTool dialogShowFgPlato={dialogShowFgPlato} closeDialogFgPlato={() => setDialogShowFgPlato(false)} />
    </div>
  )
}

export default Home
