const _ = require('underscore')

const equivSugar = fermentable => (fermentable.amount / 1000) * (fermentable.yield / 100)

const colorArray = () => ['FFE699', 'FFD878', 'FFCA5A', 'FFBF42', 'FBB123', 'F8A600', 'F39C00', 'EA8F00', 'E58500', 'DE7C00', 'D77200', 'CF6900', 'CB6200', 'C35900', 'BB5100', 'B54C00', 'B04500', 'A63E00', 'A13700', '9B3200', '952D00', '8E2900', '882300', '821E00', '7B1A00', '771900', '701400', '6A0E00', '660D00', '5E0B00', '5A0A02', '600903', '520907', '4C0505', '470606', '440607', '3F0708', '3B0607', '3A070B', '36080A']

const calcSugars = (fermentables) => {
  const sugarEquivalents = {
    totalSugars: 0,
    mashedSugars: 0,
    nonMashedSugars: 0,
    preBoilSugars: 0,
    preBoilMashed: 0,
    preBoilNonMashed: 0
  }
  fermentables.forEach((fermentable) => {
    sugarEquivalents.totalSugars += equivSugar(fermentable)
    if (fermentable.type === 'Sugar' || fermentable.type === 'Dry Extract' || fermentable.type === 'Extract') {
      sugarEquivalents.nonMashedSugars += equivSugar(fermentable)
    } else if (fermentable.use === 'add_to_mash') {
      sugarEquivalents.mashedSugars += equivSugar(fermentable)
    } else {
      sugarEquivalents.mashedSugars += 0
    }

    /* Sugars added after boil, to compute pre-boil gravity. Impact on IBU.  */
    if (fermentable.add_after_boil !== 'TRUE' && fermentable.use === 'add_to_mash') {
      sugarEquivalents.preBoilSugars += equivSugar(fermentable)
      if (fermentable.type === 'Sugar' || fermentable.type === 'Dry Extract' || fermentable.type === 'Extract') {
        sugarEquivalents.preBoilNonMashed += equivSugar(fermentable)
      } else {
        sugarEquivalents.preBoilMashed += equivSugar(fermentable)
      }
    }
  })

  return sugarEquivalents
}

const calcGravityUnits = (fermentables, volume, efficiency) => {
  const total = (((383.89 * calcSugars(fermentables).mashedSugars) / volume) * (efficiency / 100)) +
    ((383.89 * calcSugars(fermentables).nonMashedSugars) / volume)
  return total
}

const calcPreBoilGravityUnits = (fermentables, volume, efficiency) => {
  const total = (((383.89 * calcSugars(fermentables).preBoilMashed) / volume) * (efficiency / 100)) +
    ((383.89 * calcSugars(fermentables).preBoilNonMashed) / volume)
  return total
}

const calcPreBoilGravity = (fermentables, volume, efficiency) => {
  const total = 1 + (calcPreBoilGravityUnits(fermentables, volume, efficiency) / 1000)
  return total
}

const calcOriginalGravity = (recipe) => {
  const total = 1 +
    (calcGravityUnits(recipe.fermentables, recipe.batch_size, recipe.efficiency) / 1000)
  return total
}

const calcIbus = function (recipe) {
  /*      #Tinseth method
      #IBUs = decimal alpha acid utilization * mg/l of added alpha acids

      #mg/l of added alpha acids = decimal AA rating * grams hops * 1000 / liters of wort
      #Decimal Alpha Acid Utilization = Bigness Factor * Boil Time Factor
      #Bigness factor = 1.65 * 0.000125^(wort gravity - 1)
      #Boil Time factor = 1 - e^(-0.04 * time in mins) / 4.15
      */
  let totalIbus = 0
  let ibu
  const bignessFactor = 1.65 * (Math.pow(0.000125, (calcPreBoilGravity(recipe.fermentables, recipe.batch_size, recipe.efficiency) - 1)))
  recipe.hops.forEach(function (hop) {
    const btFactor = (1 - Math.pow(2.71828182845904523536, (-0.04 * hop.time))) / 4.15
    const decimalUtil = btFactor * bignessFactor
    const mgAcid = (hop.alpha / 100) * (hop.amount * 1000) / recipe.batch_size

    if (hop.use !== 'Dry Hop') {
      ibu = mgAcid * decimalUtil
      if (hop.form === 'Pellet') {
        ibu = ibu + 0.1 * ibu
      }
    } else {
      ibu = 0
    }

    if (hop.use === 'Flame Out') {
      ibu = 0.5 * ibu
    }

    totalIbus += ibu
    hop.ibuPart = ibu
  })
  recipe.ibu = totalIbus
  return recipe
}

export function ebc (fermentables, volume) {
  /*          calcul de la couleur
          calcul du MCU pour chaque grain :
          MCU=4.23*EBC(grain)*Poids grain(Kg)/Volume(L)
          puis addition de tous les MCU
          puis calcul EBC total :
          EBC=2.939*MCU^0.6859 */
  let mcuTot = 0
  let mcu
  fermentables.forEach(function (fermentable) {
    mcu = 4.23 * fermentable.color * (fermentable.amount / 1000) / volume
    mcuTot = mcuTot + mcu
  })
  const ebc = 2.939 * Math.pow(mcuTot, 0.6859)
  return ebc
}

export function sugars (fermentables) {
  return calcSugars(fermentables)
}

export function weight (fermentables) {
  let weight = 0
  fermentables.forEach(function (fermentable) {
    if (fermentable.type === 'Grain' && fermentable.use === 'add_to_mash') {
      weight += fermentable.amount
    }
  })
  return weight
}

export function originalGravity (recipe) {
  return calcOriginalGravity(recipe)
}

export function finalGravity (recipe) {
  /* Use the highest attenuation in a list of yeasts */
  let hiAtten
  try {
    recipe.yeasts = _.sortBy(recipe.yeasts, 'attenuation')
    hiAtten = _.last(recipe.yeasts).attenuation
  } catch (e) {
    hiAtten = 75
  }

  if (hiAtten === 'undefined') {
    hiAtten = 75
  }
  hiAtten = hiAtten / 100
  const gu = calcGravityUnits(recipe.fermentables, recipe.batch_size, recipe.efficiency) * (1 - hiAtten)
  return 1 + gu / 1000
}

export function ibus (recipe) {
  return calcIbus(recipe)
}

export function gu (recipe) {
  return calcGravityUnits(recipe.fermentables, recipe.batch_size, recipe.efficiency)
}

export function preBoilGu (recipe) {
  return calcPreBoilGravityUnits(recipe.fermentables, recipe.batch_size, recipe.efficiency)
}

export function bugu (recipe) {
  if (recipe.ibu === 'undefined') {
    recipe.ibu = 0
  }
  return recipe.ibu / calcGravityUnits(recipe.fermentables, recipe.batch_size, recipe.efficiency)
}

export function alc (recipe) {
  /* ABV = 0.130((OG-1)-(FG-1))*1000 */
  return 0.130 * ((recipe.og - 1) - (recipe.fg - 1)) * 1000
}

export function preBoilCalc (coolingLossRate, boilOffRate, boilTime, volume) {
  const volPreCool = volume / (1 - coolingLossRate)
  const volPreBoil = volPreCool / (1 - (boilOffRate * boilTime / 60))
  return volPreBoil
}

export function preBoilSgCalc (gravityUnits, volume, volPreBoil) {
  const ratio = volume / volPreBoil
  const gus = gravityUnits * ratio
  const preBoilSg = 1 + (gus / 1000)
  return preBoilSg
}

export function strikeTempCalc (fudgeFactor, grainTemp, targetTemp, ratio) {
  /* Tstrike = [targetTemp + (0.4 * (Ttarget - Tgrain) / ratio)] + FF  */
  const strikeTemp = (targetTemp + (0.4 * (targetTemp - grainTemp) / ratio)) + fudgeFactor
  return strikeTemp
}

export function strikeVolCalc (grainWeight, ratio) {
  const strikeVol = grainWeight * ratio / 1000
  return strikeVol
}

export function strikeVolreCalc (fudgeFactor, grainTemp, targetTemp, grainWeight, strikeTemp) {
  /* strike vol when temp changed */
  grainWeight = grainWeight / 1000
  const strikeVol = 0.4 * grainWeight * (targetTemp - grainTemp) / (strikeTemp - targetTemp - fudgeFactor)
  return strikeVol
}

export function infusionVolCalc (grainWeight, ratio, targetTemp, mashTemp, mashVolume, strikeTemp, fudgeFactor) {
  /* Vm = Wgrain (0.4 + ratio)
  Tstrike = (Ttarget*(Vstrike+Vm) - (Vm*Tmash)) / Vstrike */
  const Vm = (grainWeight / 1000) * (0.4 + ratio)
  strikeTemp = strikeTemp - fudgeFactor
  const infuseVol = ((targetTemp * Vm) - (Vm * mashTemp)) / (strikeTemp - targetTemp)
  const newRatio = (mashVolume + infuseVol) / (grainWeight / 1000)
  return { 'newRatio': newRatio, 'infuseVol': infuseVol }
}

export function infusionTempCalc (grainWeight, ratio, targetTemp, mashTemp, mashVolume, strikeVol) {
  const Vm = (grainWeight / 1000) * (0.4 + ratio)
  const strikeTemp = ((targetTemp * (strikeVol + Vm)) - (Vm * mashTemp)) / strikeVol
  return strikeTemp
}

export function grainRetentionCalc (grainRetentionRate, grainWeight) {
  const grainRetentionVol = grainRetentionRate * grainWeight / 1000
  return grainRetentionVol
}

export function spargeVolCalc (grainRetentionVol, volPreBoil, stepsVol) {
  const spargeVol = volPreBoil - (stepsVol - grainRetentionVol)
  return spargeVol
}

export function stepsVol (steps) {
  const mashVolume = steps.reduce((accu, item) => accu + item.waterVol, 0)
  return mashVolume
}

export function grainVolumeCalc (grainWeight) {
  const grainVolume = grainWeight * 1.5 / 1000
  return grainVolume
}

export function mashVolumeCalc (grainWeight, strikeVol) {
  /* 1l water+ 500g grain = 1.325 l to saturate grain. After saturation add water vol. */
  const satGrain = grainWeight * 2 / 1000
  const volSat = satGrain * 1.325
  const waterAfterSat = strikeVol - satGrain
  const mashVolume = volSat + waterAfterSat
  return mashVolume
}

export function mashVolumeLastStepCalc (mashVolumeStrike, strikeVol, mashVol) {
  const mashVolumeLastStep = mashVolumeStrike + mashVol - strikeVol
  return mashVolumeLastStep
}

export function checkBiab (steps) {
  let infusionSteps = 0
  let i
  for (i = 0; i < steps.length; i += 1) {
    if (steps[i].type === 'Infusion') {
      infusionSteps += 1
    }
  }
  if (infusionSteps > 1) {
    return false
  } else {
    return true
  }
}

export function biabCalc (preBoilVol, grainWeight, config, target) {
  const strikeVol = preBoilVol + grainRetentionCalc(config.GrainRetention, grainWeight)
  const biabRatio = strikeVol / (grainWeight / 1000)
  const biabTemp = strikeTempCalc(parseFloat(config.FudgeFactor), parseFloat(config.GrainTemp), target, biabRatio)
  const biabVolume = mashVolumeCalc(grainWeight, strikeVol)
  return {
    targetTemp: target,
    strikeVol: strikeVol,
    ratio: biabRatio,
    temp: biabTemp,
    mashVolume: biabVolume
  }
}

export function scaleIngredients (ratio, recipe) {
  const newRecipe = { ...recipe }
  const scale = (item) => {
    const newItem = { ...item }
    newItem.amount = item.amount * ratio
    return newItem
  }
  newRecipe.fermentables = recipe.fermentables.map(item => scale(item))
  newRecipe.hops = recipe.hops.map(item => scale(item))
  newRecipe.miscs = recipe.miscs.map(item => scale(item))
  return newRecipe
}

export function ingRatio (ingredients, amount) {
  if (amount === 0) {
    return 0
  }
  let weight = 0
  ingredients.forEach(function (ingredient) {
    weight += ingredient.amount
  })
  const ratio = amount / weight
  return ratio
}

export function colorHtml (ebc) {
  let colorId
  const srm = Math.round(parseFloat(ebc) / 1.97)
  if (srm <= 1) {
    colorId = '#' + colorArray()[0]
  } else if (srm >= 30) {
    colorId = '#' + colorArray()[30]
  } else {
    colorId = '#' + colorArray()[srm - 1]
  }

  return colorId
}

export function stepsCalc (classic, steps, grainWeight, config) {
  function processFirst () {
    const firstStep = {}
    firstStep.waterVol = strikeVolCalc(parseFloat(grainWeight), classic.baseRatio)
    firstStep.waterTemp = strikeTempCalc(
      parseFloat(config.FudgeFactor),
      parseFloat(config.GrainTemp),
      parseFloat(steps[0].step_temp),
      classic.baseRatio
    )
    firstStep.ratio = classic.baseRatio
    firstStep.temp = parseFloat(steps[0].step_temp)
    firstStep.type = 'Infusion'
    firstStep.name = steps[0].name
    firstStep.time = steps[0].step_time
    return [firstStep]
  }

  function processInfusion (step, listStep, index) {
    let waterTemp = 0
    try {
      waterTemp = classic.steps[index + 1].waterTemp
    } catch (e) {
      waterTemp = 90
    }

    const beforeStep = [...listStep].pop()
    const mashTemp = beforeStep.temp
    const { ratio } = beforeStep
    const mashVolume = listStep.reduce((accu, item) => accu + item.waterVol, 0)

    const calculatedInfusion = infusionVolCalc(
      grainWeight,
      ratio,
      step.step_temp,
      mashTemp,
      mashVolume,
      waterTemp,
      parseFloat(config.FudgeFactor)
    )
    return [{
      waterVol: calculatedInfusion.infuseVol,
      waterTemp,
      ratio: calculatedInfusion.newRatio,
      temp: parseFloat(step.step_temp),
      type: 'Infusion',
      name: step.name,
      time: step.step_time
    }]
  }

  function processTemp (step, listStep) {
    const beforeStep = [...listStep].pop()
    return [{
      waterVol: 0,
      ratio: beforeStep.ratio,
      temp: parseFloat(step.step_temp),
      type: 'Temperature',
      name: step.name,
      time: step.step_time
    }]
  }

  function processSteps (firstStep) {
    return steps.slice(1).reduce(
      (accu, step, index) =>
        accu.concat(step.type === 'Infusion' ? processInfusion(step, accu, index) : processTemp(step, accu))
      , [...firstStep]
    )
  }

  const allSteps = processSteps(processFirst())
  // console.log(allSteps);
  return allSteps
}

export function processSparge (listSteps, config, grainWeight, preBoilVolume) {
  const grainRetentionVol = grainRetentionCalc(config.GrainRetention, grainWeight)
  const volSteps = listSteps.reduce((accu, item) => accu + item.waterVol, 0)
  const spargeVol = spargeVolCalc(grainRetentionVol, preBoilVolume, volSteps)
  return spargeVol
}
