const fs = require('fs')
const path = require('path')
const XML = require('pixl-xml')
const nanoid = require('nanoid')
const _ = require('underscore')

function walkRecipes (dir) {
  return fs.readdirSync(dir).reduce((list, file) => {
    const name = path.join(dir, file)
    const isDir = fs.statSync(name).isDirectory()
    return list.concat(isDir ? walkRecipes(name) : [name])
  }, [])
}

function parseAll (f, dirStorage) {
  // First we need a list of paths.
  // f is the function used to read each directory inside dirStorage.
  const listFiles = f(dirStorage)
  return listFiles.reduce((list, filePath) => {
    const recipeObject = parseFile(filePath)
    if (typeof recipeObject !== 'undefined') {
      recipeObject.path = filePath
      return list.concat([recipeObject])
    }
    return list
  }, [])
}

function parseFile (filePath) {
  const rawXml = fs.readFileSync(filePath, 'utf8')
  try {
    const rawObject = XML.parse(rawXml, { lowerCase: true }).recipe
    return rawObject
  } catch (err) {
    console.log('XML parser error : ' + err)
  }
}

export function recipesList (recipesDir) {
  // parse all recipes recursively
  const recipes = parseAll(walkRecipes, recipesDir)

  // cleaning ingredients arrays.
  function flatten (recipe) {
    // recipe.fermentables.fermentable : [] ---> recipe.fermentables : []
    // if only 1 fermentable,
    //
    // recipe.fermentables.fermentable : fermentable ---> recipe.fermentables : [fermentable]
    // if 0 fermentable ---> []
    const fermentablesArray = []
    if (recipe.fermentables.fermentable instanceof Array) {
      recipe.fermentables.fermentable.map((item) => fermentablesArray.push(item))
    } else {
      fermentablesArray.push(recipe.fermentables.fermentable)
    }
    recipe.fermentables = (fermentablesArray[0] == null) ? [] : fermentablesArray

    const hopsArray = []
    if (recipe.hops.hop instanceof Array) {
      recipe.hops.hop.map((item) => hopsArray.push(item))
    } else {
      hopsArray.push(recipe.hops.hop)
    }
    recipe.hops = (hopsArray[0] == null) ? [] : hopsArray

    const miscsArray = []
    if (recipe.miscs.misc instanceof Array) {
      recipe.miscs.misc.map((item) => miscsArray.push(item))
    } else {
      miscsArray.push(recipe.miscs.misc)
    }
    recipe.miscs = (miscsArray[0] == null) ? [] : miscsArray

    const yeastsArray = []
    if (recipe.yeasts.yeast instanceof Array) {
      recipe.yeasts.yeast.map((item) => yeastsArray.push(item))
    } else {
      yeastsArray.push(recipe.yeasts.yeast)
    }
    recipe.yeasts = (yeastsArray[0] == null) ? [] : yeastsArray

    const stepsArray = []
    if (recipe.mash.mash_steps.mash_step instanceof Array) {
      recipe.mash.mash_steps.mash_step.map((item) => stepsArray.push(item))
    } else {
      stepsArray.push(recipe.mash.mash_steps.mash_step)
    }
    recipe.mash.mash_steps = (stepsArray[0] == null) ? [] : stepsArray

    return recipe
  }

  function fixAmounts (recipe) {
    const toGrams = (item) => { item.amount *= 1000; return item }
    recipe.fermentables = recipe.fermentables.map(toGrams)
    recipe.hops = recipe.hops.map(toGrams)
    recipe.miscs = recipe.miscs.map(toGrams)
    return recipe
  }

  function fixNumbers (recipe) {
    recipe.batch_size = parseFloat(recipe.batch_size)
    recipe.efficiency = parseFloat(recipe.efficiency)
    recipe.boil_time = parseFloat(recipe.boil_time)
    recipe.fermentables.map((item) => {
      item.amount = parseFloat(item.amount)
      item.yield = parseFloat(item.yield)
      item.color = parseFloat(item.color)
    })
  }

  function fixColor (recipe) {
    const toEbc = (item) => { item.color = item.color * 1.97; return item }
    recipe.fermentables = recipe.fermentables.map(toEbc)
    return recipe
  }

  function fixFermentableUse (recipe) {
    const addDefault = (item) => {
      if (item.add_after_boil === 'TRUE') {
        item.use = 'add_to_flameout'
        return item
      }
      item.use = 'add_to_mash'
      return item
    }
    const checkUse = item => item.use ? item : addDefault(item)
    recipe.fermentables = recipe.fermentables.map(checkUse)
    return recipe
  }

  function sortHops (recipe) {
    const groupHops = use => recipe.hops.filter(hop => hop.use === use)
    recipe.hops = [].concat(
      _.sortBy(groupHops('Mash'), 'time').reverse(),
      _.sortBy(groupHops('First Wort'), 'time').reverse(),
      _.sortBy(groupHops('Boil'), 'time').reverse(),
      _.sortBy(groupHops('Aroma'), 'time').reverse(),
      _.sortBy(groupHops('Dry Hop'), 'time').reverse()
    )
    return recipe
  }

  function addId (recipe) {
    const appendId = (item) => {
      const newItem = item
      newItem.id = nanoid()
      newItem.nameId = nanoid()
      return newItem
    }
    recipe.fermentables.map(fermentable => appendId(fermentable))
    recipe.hops.map(hop => appendId(hop))
    recipe.miscs.map(misc => appendId(misc))
    recipe.yeasts.map(yeast => appendId(yeast))
  }

  recipes.map(flatten)
  recipes.map(fixAmounts)
  recipes.map(fixNumbers)
  recipes.map(fixColor)
  recipes.map(fixFermentableUse)
  recipes.map(addId)
  recipes.map((recipe) => {
    const newRecipe = recipe
    newRecipe.id = nanoid()
    return newRecipe
  })
  recipes.map(sortHops)

  return recipes
}
