import * as beerCalc from './beercalc'
const _ = require('underscore')
const nanoid = require('nanoid')

export const profileCalc = (recipeJson) => {
  const recipe = { ...recipeJson }
  recipe.ebc = Math.round(beerCalc.ebc(recipe.fermentables, recipe.batch_size))
  recipe.ibu = Math.round(beerCalc.ibus(recipe).ibu)
  recipe.og = Math.round(beerCalc.originalGravity(recipe) * 1000) / 1000
  recipe.fg = Math.round(beerCalc.finalGravity(recipe) * 1000) / 1000
  recipe.alc = Math.round(beerCalc.alc(recipe) * 10) / 10
  recipe.bugu = Math.round(beerCalc.bugu(recipe) * 10) / 10

  recipe.preboilGravity = beerCalc.preBoilSgCalc(
    beerCalc.preBoilGu(recipe),
    recipe.batch_size,
    recipe.preboilVolume
  )
  recipe.grainWeight = beerCalc.weight(recipe.fermentables)
  recipe.grainVolume = beerCalc.grainVolumeCalc(recipe.grainWeight)
  return recipe
}

export const preboilVolume = (recipeJson, config) => {
  const recipe = { ...recipeJson }
  recipe.preboilVolume = beerCalc.preBoilCalc(
    config.General.CoolingLoss / 100,
    config.General.BoilOffRate / 100,
    recipe.boil_time, recipe.batch_size
  )
  return recipe
}

export const preboilGravity = (recipeJson) => {
  const recipe = { ...recipeJson }
  recipe.preboilGravity = beerCalc.preBoilSgCalc(
    beerCalc.preBoilGu(recipe),
    recipe.batch_size,
    recipe.preboilVolume
  )
  return recipe
}

export const calcAll = (recipeJson, config) => {
  if (recipeJson.batch_size === '0' || recipeJson.batch_size === '' || recipeJson.batch_size === undefined) {
    return recipeJson
  } else {
    return preboilGravity(preboilVolume(profileCalc(recipeJson), config))
  }
}

export const biab = (recipeJson, config, target) =>
  beerCalc.biabCalc(recipeJson.preboilVolume, recipeJson.grainWeight, config.General, target)

export const classic = (recipeJson, config) => {
  const initClassic = { baseRatio: 3, steps: [] }
  const calculatedClassic = {}
  calculatedClassic.steps = beerCalc.stepsCalc(initClassic, recipeJson.mash.mash_steps, recipeJson.grainWeight, config.General)
  calculatedClassic.stepsVol = beerCalc.stepsVol(calculatedClassic.steps)
  calculatedClassic.spargeVol = beerCalc.processSparge(calculatedClassic.steps, config.General, recipeJson.grainWeight, recipeJson.preboilVolume)
  calculatedClassic.mashVolume = beerCalc.mashVolumeCalc(recipeJson.grainWeight, calculatedClassic.steps[0].waterVol)
  calculatedClassic.mashVolumeLastStep = (calculatedClassic.mashVolume + calculatedClassic.stepsVol) - calculatedClassic.steps[0].waterVol
  return calculatedClassic
}

export const scaleAndCalc = (recipeJson, ratio, config) => {
  if (ratio !== 0 || ratio !== undefined || ratio !== null) {
    const recipe = beerCalc.scaleIngredients(ratio, recipeJson)
    return calcAll(recipe, config)
  }
  return recipeJson
}

export const updateList = (itemList, updatedItem) =>
  itemList.map(item => ((item.id === updatedItem.id) ? updatedItem : item))

export const deleteItem = (itemList, itemToDel) =>
  itemList.filter(item => (item.id !== itemToDel.id))

export const historyAppend = (history, newItem) => {
  if ([...history].pop().historyMessage === newItem.historyMessage) {
    return history.slice(0, -1).concat([newItem])
  }
  return history.concat([newItem])
}

export const sortRecipe = recipe => {
  const categorizeFermentables = use => recipe.fermentables.filter(fermentable => fermentable.use === use)
  const sortedFermentables = [].concat(
    _.chain(categorizeFermentables('add_to_mash')).sortBy('name').sortBy('amount').reverse().value(),
    _.chain(categorizeFermentables('add_to_boil')).sortBy('name').sortBy('amount').reverse().value(),
    _.chain(categorizeFermentables('add_to_flameout')).sortBy('name').sortBy('amount').reverse().value(),
    _.chain(categorizeFermentables('add_to_fermentation')).sortBy('name').sortBy('amount').reverse().value(),
    _.chain(categorizeFermentables('add_to_package')).sortBy('name').sortBy('amount').reverse().value()
  )
  const categorizeHops = use => recipe.hops.filter(hop => hop.use === use)
  const sortedHops = [].concat(
    _.chain(categorizeHops('Mash')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeHops('First Wort')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeHops('Boil')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeHops('Flame Out')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeHops('Dry Hop')).sortBy('name').sortBy('time').reverse().value()
  )
  const categorizeMiscs = use => recipe.miscs.filter(misc => misc.use === use)
  const sortedMiscs = [].concat(
    _.chain(categorizeMiscs('Mash')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeMiscs('Boil')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeMiscs('Primary')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeMiscs('Secondary')).sortBy('name').sortBy('time').reverse().value(),
    _.chain(categorizeMiscs('Bottling')).sortBy('name').sortBy('time').reverse().value()
  )
  if (!recipe.hasOwnProperty('snapshots')) {
    recipe.snapshots = []
  }
  recipe.fermentables = sortedFermentables
  recipe.hops = sortedHops
  recipe.miscs = sortedMiscs
  return recipe
}

export const sortListByName = list => {
  const sortedList = [].concat(
    _.sortBy(list, function (o) { return o.name ? o.name.toLowerCase() : '' })
  )
  return sortedList
}

export const sortListByBrewer = list => {
  const sortedList = [].concat(
    _.chain(list)
      .sortBy(function (o) { return o.name ? o.name.toLowerCase() : '' })
      .sortBy(function (o) { return o.brewer ? o.brewer.toLowerCase() : '' })
      .value()
  )
  return sortedList
}

export const sortRecipesList = (list, option) => {
  if (option === 'name') {
    return sortListByName(list)
  }
  return sortListByBrewer(list)
}

export const takeSnapshot = (recipe) => {
  const date = new Date()
  const dateString = `${date.toLocaleDateString()} - ${date.toLocaleTimeString()}`
  const title = 'Automatically generated snapshot'
  return { dateString, title, recipe, id: nanoid() }
}

export const cleanRecipe = recipe => {
  const cleanHop = hop => {
    const { version, notes, origin, type, display_amount, inventory, display_time, ...updatedHop } = hop
    return updatedHop
  }
  const cleanFermentable = fermentable => {
    const { version, origin, diastatic_power, protein, max_in_batch, ibu_gal_per_lb, display_amount, inventory, potential, display_color, extract_substitute, notes, supplier, moisture, coarse_fine_diff, ...updatedFermentable } = fermentable
    return updatedFermentable
  }
  const cleanYeast = yeast => {
    const { version, amount_is_weight, min_temperature, max_temperature, flocculation, notes, best_for, max_reuse, times_cultured, add_to_secondary, display_amount, disp_min_temp, disp_max_temp, inventory, culture_date, ...updatedYeast } = yeast
    return updatedYeast
  }
  const cleanStyle = style => {
    return { name: style.name }
  }

  delete recipe.version
  delete recipe.asst_brewer
  delete recipe.equipment
  delete recipe.taste_note
  delete recipe.taste_rating
  delete recipe.carbonation
  delete recipe.fermentation_stages
  delete recipe.primary_age
  delete recipe.primary_temp
  delete recipe.secondary_age
  delete recipe.secondary_temp
  delete recipe.tertiary_age
  delete recipe.age
  delete recipe.age_temp
  delete recipe.carbonation_used
  delete recipe.priming_sugar_name
  delete recipe.priming_sugar_equiv
  delete recipe.keg_priming_factor
  delete recipe.carbonation_temp
  delete recipe.display_carb_temp
  delete recipe.date
  delete recipe.calories
  delete recipe.display_batch_size
  delete recipe.display_boil_size
  delete recipe.display_og
  delete recipe.display_fg
  delete recipe.display_primary_temp
  delete recipe.display_secondary_temp
  delete recipe.display_tertiary_temp
  delete recipe.display_age_temp
  delete recipe.taste_notes

  recipe.hops = recipe.hops.map(cleanHop)
  recipe.fermentables = recipe.fermentables.map(cleanFermentable)
  recipe.yeasts = recipe.yeasts.map(cleanYeast)
  recipe.style = cleanStyle(recipe.style)

  return recipe
}
